# Guia de Integração do Formulário de Inscrição com o Website

## 🌐 Opções de Integração

### **Opção 1: Link Direto (Mais Simples)**

A forma mais simples é criar um link no seu website que redireciona para o formulário.

#### No seu website HTML:

```html
<!-- Botão de inscrição -->
<a href="https://seu-sistema.com/inscrever-se" 
   target="_blank" 
   class="btn btn-primary btn-lg">
    <i class="bi bi-person-plus"></i> Inscrever-se Agora
</a>

<!-- Ou como seção na página -->
<section class="inscricao-section">
    <h2>Inscrições Abertas</h2>
    <p>Preencha o formulário online e aguarde a aprovação da administração.</p>
    <a href="https://seu-sistema.com/inscrever-se" 
       target="_blank" 
       class="btn btn-primary">
        Acessar Formulário de Inscrição
    </a>
</section>
```

#### Vantagens:
- ✅ Muito simples de implementar
- ✅ Não requer modificações no código
- ✅ Funciona em qualquer website

#### Desvantagens:
- ❌ O usuário sai do seu website
- ❌ Design diferente do seu site

---

### **Opção 2: Iframe (Integrado no Site)**

O formulário aparece dentro do seu website usando um iframe.

#### No seu website HTML:

```html
<!-- Seção de inscrição -->
<section class="inscricao-section" style="padding: 40px 0;">
    <div class="container">
        <h2 class="text-center mb-4">Formulário de Inscrição</h2>
        
        <!-- Iframe com o formulário -->
        <iframe 
            src="https://seu-sistema.com/inscrever-se" 
            width="100%" 
            height="1200px" 
            frameborder="0"
            style="border-radius: 10px; box-shadow: 0 4px 6px rgba(0,0,0,0.1);">
        </iframe>
    </div>
</section>
```

#### Versão Responsiva (Recomendada):

```html
<section class="inscricao-section" style="padding: 40px 0;">
    <div class="container">
        <h2 class="text-center mb-4">Formulário de Inscrição Online</h2>
        
        <div class="iframe-container" style="position: relative; width: 100%; padding-bottom: 150%; overflow: hidden;">
            <iframe 
                src="https://seu-sistema.com/inscrever-se" 
                style="position: absolute; top: 0; left: 0; width: 100%; height: 100%; border: none; border-radius: 10px;"
                frameborder="0"
                allowfullscreen>
            </iframe>
        </div>
    </div>
</section>

<style>
    @media (max-width: 768px) {
        .iframe-container {
            padding-bottom: 200% !important; /* Mais altura no mobile */
        }
    }
</style>
```

#### Vantagens:
- ✅ Formulário aparece no seu site
- ✅ Usuário não sai da página
- ✅ Design pode ser customizado

#### Desvantagens:
- ❌ Pode ter problemas de responsividade
- ❌ Alguns navegadores podem bloquear iframes

---

### **Opção 3: Popup/Modal (Elegante)**

Abre o formulário em uma janela popup ou modal.

#### No seu website HTML:

```html
<!-- Botão que abre o popup -->
<button onclick="abrirFormularioInscricao()" class="btn btn-primary btn-lg">
    Inscrever-se Agora
</button>

<script>
function abrirFormularioInscricao() {
    // Abrir em nova janela
    window.open(
        'https://seu-sistema.com/inscrever-se',
        'FormularioInscricao',
        'width=900,height=800,scrollbars=yes,resizable=yes'
    );
}

// Ou usar modal do Bootstrap (se você usa Bootstrap)
function abrirModalInscricao() {
    const modal = document.createElement('div');
    modal.className = 'modal fade';
    modal.id = 'modalInscricao';
    modal.innerHTML = `
        <div class="modal-dialog modal-xl">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Formulário de Inscrição</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body" style="padding: 0;">
                    <iframe 
                        src="https://seu-sistema.com/inscrever-se" 
                        width="100%" 
                        height="800px" 
                        frameborder="0">
                    </iframe>
                </div>
            </div>
        </div>
    `;
    document.body.appendChild(modal);
    const bsModal = new bootstrap.Modal(modal);
    bsModal.show();
    
    // Remover modal quando fechar
    modal.addEventListener('hidden.bs.modal', function() {
        modal.remove();
    });
}
</script>
```

#### Vantagens:
- ✅ Experiência de usuário elegante
- ✅ Não interfere no layout principal
- ✅ Fácil de fechar

#### Desvantagens:
- ❌ Requer JavaScript
- ❌ Popups podem ser bloqueados

---

### **Opção 4: API REST (Avançado)**

Criar um endpoint API para receber dados do website via AJAX.

#### 1. Criar rota API (no Laravel):

```php
// routes/api.php
Route::post('/inscricao', [App\Http\Controllers\PublicInscricaoController::class, 'storeApi'])
    ->name('api.inscricao.store');
```

#### 2. Criar método no Controller:

```php
// app/Http/Controllers/PublicInscricaoController.php
public function storeApi(Request $request)
{
    // Mesma validação do método store()
    $validator = Validator::make($request->all(), [
        'name' => 'required|string|max:255',
        'telefone_encarregado' => 'required|string',
        // ... outros campos
    ]);

    if ($validator->fails()) {
        return response()->json([
            'success' => false,
            'errors' => $validator->errors()
        ], 422);
    }

    // Processar inscrição (mesmo código do store())
    // ...

    return response()->json([
        'success' => true,
        'message' => 'Inscrição submetida com sucesso!',
        'data' => [
            'codigo_estudante' => $aluno->codigo_estudante
        ]
    ]);
}
```

#### 3. No seu website HTML:

```html
<form id="formInscricao" onsubmit="submeterInscricao(event)">
    <!-- Seus campos customizados -->
    <input type="text" name="name" required>
    <input type="text" name="telefone_encarregado" required>
    <!-- ... outros campos -->
    
    <button type="submit">Submeter</button>
</form>

<script>
async function submeterInscricao(event) {
    event.preventDefault();
    
    const form = event.target;
    const formData = new FormData(form);
    
    try {
        const response = await fetch('https://seu-sistema.com/api/inscricao', {
            method: 'POST',
            body: formData,
            headers: {
                'X-Requested-With': 'XMLHttpRequest'
            }
        });
        
        const data = await response.json();
        
        if (data.success) {
            alert('Inscrição submetida com sucesso!');
            form.reset();
        } else {
            alert('Erro: ' + JSON.stringify(data.errors));
        }
    } catch (error) {
        alert('Erro ao submeter inscrição: ' + error.message);
    }
}
</script>
```

#### Vantagens:
- ✅ Controle total sobre o design
- ✅ Integração perfeita
- ✅ Pode customizar completamente

#### Desvantagens:
- ❌ Requer desenvolvimento adicional
- ❌ Mais complexo de manter

---

## 🎨 Personalização do Formulário

### Alterar URL do Formulário

Se você quiser usar uma URL personalizada no seu website:

1. **Criar rota personalizada** (opcional):
```php
// routes/web.php
Route::get('/matriculas', [App\Http\Controllers\PublicInscricaoController::class, 'showForm'])
    ->name('public.inscricao.custom');
```

2. **Usar no website**:
```html
<a href="https://seu-sistema.com/matriculas">Inscrever-se</a>
```

### Adicionar Logo da Escola

Edite `resources/views/layouts/public.blade.php`:

```blade
<div class="public-header">
    <img src="{{ asset('storage/logo.png') }}" alt="Logo" style="max-height: 80px; margin-bottom: 15px;">
    <h1>{{ \App\Models\Configuracao::get('escola', 'Colégio Premium') }}</h1>
    <p>Formulário de Inscrição Online</p>
</div>
```

### Personalizar Cores

As cores são definidas pelo tema ativo no sistema. Você pode alterar em:
- Admin → Configurações → Temas

Ou editar diretamente em `resources/views/layouts/public.blade.php`:

```css
:root {
    --cor-principal: #seu-codigo-cor; /* Ex: #0066cc */
}
```

---

## 📱 Exemplo Completo para WordPress

Se o seu website é WordPress, adicione este código no tema:

```php
<?php
// functions.php ou no editor de temas
function adicionar_botao_inscricao() {
    ?>
    <div class="inscricao-widget">
        <h3>Inscrições Abertas</h3>
        <p>Preencha o formulário online</p>
        <a href="https://seu-sistema.com/inscrever-se" 
           target="_blank" 
           class="btn-inscricao">
            Inscrever-se
        </a>
    </div>
    <?php
}
add_shortcode('inscricao', 'adicionar_botao_inscricao');
```

Use no conteúdo: `[inscricao]`

---

## 🔒 Segurança e CORS

Se você usar a Opção 4 (API), pode precisar configurar CORS:

```php
// config/cors.php ou app/Http/Middleware/Cors.php
'allowed_origins' => [
    'https://seu-website.com',
    'https://www.seu-website.com',
],
```

---

## ✅ Recomendação

Para a maioria dos casos, recomendo a **Opção 1 (Link Direto)** ou **Opção 2 (Iframe)**:

- **Link Direto**: Se você quer simplicidade máxima
- **Iframe**: Se você quer manter o usuário no seu site

A **Opção 4 (API)** só é necessária se você precisar de integração muito customizada.

---

## 🆘 Suporte

Se precisar de ajuda com a integração, verifique:
1. A URL do sistema está correta?
2. O formulário está acessível publicamente?
3. Há algum firewall bloqueando?

Teste a URL diretamente: `https://seu-sistema.com/inscrever-se`
