<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\File;

class AplicarAntigoAcordoOrtografico extends Command
{
    protected $signature = 'portugues:aplicar-antigo-acordo {--dry-run}';
    protected $description = 'Aplica o português do antigo acordo ortográfico em todas as views';

    // Mapeamento de substituições (novo → antigo)
    private $substituicoes = [
        // Ações
        'Ações' => 'Acções',
        'ações' => 'acções',
        'Ação' => 'Acção',
        'ação' => 'acção',
        
        // Ativo/Inativo
        'Ativo' => 'Activo',
        'ativo' => 'activo',
        'Ativa' => 'Activa',
        'ativa' => 'activa',
        'Inativo' => 'Inactivo',
        'inativo' => 'inactivo',
        'Inativa' => 'Inactiva',
        'inativa' => 'inactiva',
        
        // Direção
        'Direção' => 'Direcção',
        'direção' => 'direcção',
        'Direções' => 'Direcções',
        'direções' => 'direcções',
        
        // Exceção
        'Exceção' => 'Excepção',
        'exceção' => 'excepção',
        'Exceções' => 'Excepções',
        'exceções' => 'excepções',
        
        // Correção
        'Correção' => 'Correcção',
        'correção' => 'correcção',
        'Correções' => 'Correcções',
        'correções' => 'correcções',
        
        // Seção
        'Seção' => 'Secção',
        'seção' => 'secção',
        'Seções' => 'Secções',
        'seções' => 'secções',
        
        // Redação
        'Redação' => 'Redacção',
        'redação' => 'redacção',
        'Redações' => 'Redacções',
        'redações' => 'redacções',
    ];

    // Palavras que NÃO devem ser alteradas (já estão corretas)
    private $excecoes = [
        'educação', 'Educação',
        'avaliação', 'Avaliação',
        'avaliações', 'Avaliações',
        'configuração', 'Configuração',
        'configurações', 'Configurações',
        'informação', 'Informação',
        'informações', 'Informações',
        'comunicação', 'Comunicação',
        'aplicação', 'Aplicação',
        'publicação', 'Publicação',
    ];

    public function handle()
    {
        $dryRun = $this->option('dry-run');
        
        if ($dryRun) {
            $this->info('Modo DRY-RUN: Nenhuma alteração será feita.');
        }
        
        $viewsPath = resource_path('views');
        $files = File::allFiles($viewsPath);
        
        $totalArquivos = 0;
        $totalAlteracoes = 0;
        
        foreach ($files as $file) {
            if ($file->getExtension() !== 'blade.php') {
                continue;
            }
            
            $content = File::get($file->getPathname());
            $originalContent = $content;
            $alteracoesArquivo = 0;
            
            // Aplicar substituições (ordem: mais específicas primeiro)
            foreach ($this->substituicoes as $novo => $antigo) {
                // Contar ocorrências antes
                $countAntes = substr_count($content, $novo);
                
                if ($countAntes > 0) {
                    // Verificar se não está em uma exceção
                    $emExcecao = false;
                    foreach ($this->excecoes as $excecao) {
                        if (stripos($content, $excecao) !== false && stripos($excecao, $novo) !== false) {
                            // Se a palavra está dentro de uma exceção, pular
                            continue 2;
                        }
                    }
                    
                    // Fazer substituição simples (não usar word boundary para evitar problemas)
                    $content = str_replace($novo, $antigo, $content);
                    $alteracoesArquivo += $countAntes;
                }
            }
            
            if ($alteracoesArquivo > 0) {
                $totalArquivos++;
                $totalAlteracoes += $alteracoesArquivo;
                
                $relativePath = str_replace(base_path() . DIRECTORY_SEPARATOR, '', $file->getPathname());
                
                if ($dryRun) {
                    $this->line("  [DRY-RUN] {$relativePath}: {$alteracoesArquivo} alterações");
                } else {
                    File::put($file->getPathname(), $content);
                    $this->line("  ✓ {$relativePath}: {$alteracoesArquivo} alterações");
                }
            }
        }
        
        $this->newLine();
        if ($dryRun) {
            $this->info("Total: {$totalArquivos} arquivos com {$totalAlteracoes} alterações (DRY-RUN)");
            $this->warn("Execute sem --dry-run para aplicar as alterações.");
        } else {
            $this->info("✓ Concluído: {$totalArquivos} arquivos alterados com {$totalAlteracoes} substituições");
        }
        
        return 0;
    }
}
