<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Professor;

class DiagnosticarAcessoProfessor extends Command
{
    protected $signature = 'professor:diagnosticar-acesso {codigo?}';
    protected $description = 'Diagnostica problemas de acesso de professores';

    public function handle()
    {
        $codigo = $this->argument('codigo');
        
        if ($codigo) {
            $professor = Professor::where('numero_funcionario', $codigo)->first();
            if (!$professor) {
                $this->error("Professor com código '{$codigo}' não encontrado.");
                return 1;
            }
            $this->diagnosticarProfessor($professor);
        } else {
            // Diagnosticar todos os professores
            $professores = Professor::with('user')->limit(10)->get();
            $this->info("Diagnosticando primeiros 10 professores...\n");
            
            foreach ($professores as $professor) {
                $this->diagnosticarProfessor($professor);
                $this->line("---");
            }
        }
        
        return 0;
    }
    
    private function diagnosticarProfessor(Professor $professor)
    {
        $this->info("Professor: {$professor->numero_funcionario}");
        
        if (!$professor->user) {
            $this->error("  ✗ SEM USER associado!");
            return;
        }
        
        $user = $professor->user;
        $this->line("  User ID: {$user->id}");
        $this->line("  Nome: {$user->name}");
        $this->line("  Email: {$user->email}");
        $this->line("  is_active: " . var_export($user->is_active, true));
        $this->line("  Tipo: {$user->tipo}");
        
        // Verificar roles
        $roles = $user->roles->pluck('slug')->toArray();
        $this->line("  Roles: " . implode(', ', $roles ?: ['NENHUMA']));
        
        if (!in_array('professor', $roles)) {
            $this->error("  ✗ NÃO TEM ROLE 'professor'!");
            $this->line("  → Adicionando role 'professor'...");
            
            $roleProfessor = \App\Models\Role::where('slug', 'professor')->first();
            if ($roleProfessor) {
                if (!$user->roles->contains($roleProfessor->id)) {
                    $user->roles()->attach($roleProfessor->id);
                    $this->info("  ✓ Role 'professor' adicionada!");
                }
            } else {
                $this->error("  ✗ Role 'professor' não existe no banco!");
            }
        } else {
            $this->info("  ✓ Tem role 'professor'");
        }
        
        // Verificar se pode acessar
        $podeAcessar = $user->is_active && in_array('professor', $roles);
        if ($podeAcessar) {
            $this->info("  ✓ PODE ACESSAR como professor");
        } else {
            $this->error("  ✗ NÃO PODE ACESSAR como professor");
            if (!$user->is_active) {
                $this->line("    - Motivo: is_active = false");
            }
            if (!in_array('professor', $roles)) {
                $this->line("    - Motivo: não tem role 'professor'");
            }
        }
    }
}
