<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\AnoLetivo;
use App\Models\Horario;
use App\Models\Turma;
use App\Models\Avaliacao;
use App\Models\Trabalho;
use App\Models\Evento;
use App\Models\Financeiro;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;

class AlunoController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
        $this->middleware('role:aluno');
    }

    /**
     * Classificar média por faixa
     */
    private function classificarMedia($media)
    {
        if ($media === null) {
            return '-';
        }

        $mediaInt = (int) $media;

        if ($mediaInt >= 0 && $mediaInt <= 9) {
            return 'Não Satisfatório';
        } elseif ($mediaInt >= 10 && $mediaInt <= 13) {
            return 'Satisfatório';
        } elseif ($mediaInt >= 14 && $mediaInt <= 16) {
            return 'Bom';
        } elseif ($mediaInt >= 17 && $mediaInt <= 18) {
            return 'Muito Bom';
        } elseif ($mediaInt >= 19 && $mediaInt <= 20) {
            return 'Excelente';
        }

        return '-';
    }

    /**
     * Dashboard do estudante
     */
    public function dashboard()
    {
        $user = auth()->user();
        $aluno = $user->aluno;
        $anoLetivoAtivo = AnoLetivo::getAnoLetivoAtual();

        if (!$aluno) {
            return redirect()->route('dashboard')
                ->withErrors(['error' => 'Seu perfil de estudante não está completo.']);
        }

        // Buscar turmas do estudante
        $turmas = collect();
        if ($anoLetivoAtivo) {
            $turmas = $aluno->turmas()
                ->wherePivot('ano_letivo_id', $anoLetivoAtivo->id)
                ->get();
        }

        // Buscar horários das turmas do estudante
        $horarios = collect();
        if ($anoLetivoAtivo && $turmas->isNotEmpty()) {
            $turmaIds = $turmas->pluck('id');
            $horarios = Horario::where('ano_letivo_id', $anoLetivoAtivo->id)
                ->whereIn('turma_id', $turmaIds)
                ->with(['turma', 'disciplina', 'professor.user'])
                ->orderBy('dia_semana')
                ->orderBy('hora_inicio')
                ->get();
        }

        // Situação financeira (para alerta de dívidas)
        // Usar o método do serviço para garantir consistência com período de graça
        $situacaoFinanceira = 'REGULAR';
        $dividas = 0;
        $isIncapacitado = false;
        
        if ($anoLetivoAtivo) {
            $financeiroService = app(\App\Services\FinanceiroService::class);
            $situacaoFinanceira = $financeiroService->getSituacaoFinanceira($aluno, $anoLetivoAtivo->id);
            $isIncapacitado = $financeiroService->isIncapacitado($aluno, $anoLetivoAtivo->id);
            
            // Considerar apenas mensalidades vencidas há mais de 4 dias (a partir do dia 6)
            // Última data sem multa: dia 5. A partir do dia 6 começa a multa.
            $dataLimite = now()->subDays(4);
            $dividas = $aluno->financeiro()
                ->where('ano_letivo_id', $anoLetivoAtivo->id)
                ->whereIn('status', ['PENDENTE', 'VENCIDO'])
                ->where('data_vencimento', '<=', $dataLimite)
                ->sum('valor');
        }

        return view('aluno.dashboard', compact(
            'aluno',
            'anoLetivoAtivo',
            'turmas',
            'horarios',
            'situacaoFinanceira',
            'dividas',
            'isIncapacitado'
        ));
    }

    /**
     * Visualizar notas do estudante
     */
    public function notas(Request $request)
    {
        $user = auth()->user();
        $aluno = $user->aluno;
        
        // Para alunos, mostrar apenas 2026 e anos anteriores (não mostrar 2027)
        // Filtrar anos letivos: apenas 2026 e anteriores
        $anosLetivos = AnoLetivo::where(function($q) {
            $q->where('ano', 'like', '%2026%')
              ->orWhere('ano', '2026')
              ->orWhere('ano', 'like', '%2025%')
              ->orWhere('ano', '2025')
              ->orWhere('ano', 'like', '%2024%')
              ->orWhere('ano', '2024')
              ->orWhere('ano', 'like', '%2023%')
              ->orWhere('ano', '2023');
        })->orderBy('ano', 'desc')->get();
        
        // Sempre usar 2026 como padrão
        $anoLetivoSelecionado = AnoLetivo::getAnoLetivoAtual();
        
        // Se houver filtro na requisição, usar esse (mas validar que não seja 2027)
        $anoLetivoId = $request->get('ano_letivo_id');
        if ($anoLetivoId) {
            $anoLetivoFiltrado = AnoLetivo::find($anoLetivoId);
            // Só permitir se não for 2027
            if ($anoLetivoFiltrado) {
                $anoRaw = $anoLetivoFiltrado->getRawOriginal('ano');
                if (strpos($anoRaw, '2027') === false && $anoRaw !== '2027') {
                    $anoLetivoAtivo = $anoLetivoFiltrado;
                } else {
                    $anoLetivoAtivo = $anoLetivoSelecionado;
                }
            } else {
                $anoLetivoAtivo = $anoLetivoSelecionado;
            }
        } else {
            $anoLetivoAtivo = $anoLetivoSelecionado;
        }

        if (!$aluno) {
            return redirect()->route('aluno.dashboard')
                ->withErrors(['error' => 'Seu perfil de estudante não está completo.']);
        }

        // Verificar se o aluno tem dívidas pendentes (apenas para o ano letivo atual)
        // Considerar apenas mensalidades vencidas há mais de 4 dias (a partir do dia 6)
        // Última data sem multa: dia 5. A partir do dia 6 começa a multa.
        $dividas = 0;
        if ($anoLetivoAtivo) {
            $dataLimite = now()->subDays(4);
            $dividas = $aluno->financeiro()
                ->where('ano_letivo_id', $anoLetivoAtivo->id)
                ->whereIn('status', ['PENDENTE', 'VENCIDO'])
                ->where('data_vencimento', '<=', $dataLimite)
                ->sum('valor');
        }

        // Só bloquear acesso se for o ano letivo atual (2026) e houver dívidas vencidas há mais de 4 dias (a partir do dia 6)
        $anoAtualAnoLetivo = AnoLetivo::getAnoLetivoAtual();
        
        if ($dividas > 0 && $anoLetivoAtivo && $anoAtualAnoLetivo && $anoLetivoAtivo->id === $anoAtualAnoLetivo->id) {
            return redirect()->route('aluno.financeiro')
                ->with('error', 'Não pode aceder às notas. Tem dívidas pendentes no valor de ' . number_format($dividas, 2, ',', '.') . ' MT. Por favor, regularize a sua situação financeira.');
        }

        // Buscar todas as avaliações do estudante
        $avaliacoes = collect();
        $trimestres = collect();
        
        if ($anoLetivoAtivo) {
            $trimestres = \App\Models\Trimestre::where('ano_letivo_id', $anoLetivoAtivo->id)
                ->orderBy('numero')
                ->get();

            $avaliacoes = $aluno->avaliacoes()
                ->where('ano_letivo_id', $anoLetivoAtivo->id)
                ->with(['disciplina', 'trimestre', 'turma'])
                ->get();

            // Organizar por disciplina e trimestre
            $avaliacoesPorDisciplina = [];
            foreach ($avaliacoes as $avaliacao) {
                $disciplinaId = $avaliacao->disciplina_id;
                $trimestreId = $avaliacao->trimestre_id;
                
                if (!isset($avaliacoesPorDisciplina[$disciplinaId])) {
                    $avaliacoesPorDisciplina[$disciplinaId] = [];
                }
                
                $avaliacoesPorDisciplina[$disciplinaId][$trimestreId] = $avaliacao;
            }
        }

        // Buscar disciplinas do estudante
        $disciplinas = collect();
        if ($anoLetivoAtivo) {
            $disciplinas = DB::table('avaliacoes')
                ->where('aluno_id', $aluno->id)
                ->where('ano_letivo_id', $anoLetivoAtivo->id)
                ->join('disciplinas', 'avaliacoes.disciplina_id', '=', 'disciplinas.id')
                ->select('disciplinas.id', 'disciplinas.nome', 'disciplinas.modelo_avaliacao')
                ->distinct()
                ->orderBy('disciplinas.nome')
                ->get();
        }

        return view('aluno.notas', compact(
            'aluno',
            'anoLetivoAtivo',
            'anosLetivos',
            'trimestres',
            'disciplinas',
            'avaliacoesPorDisciplina'
        ));
    }

    /**
     * Visualizar folheto de aproveitamento pedagógico
     */
    public function folhetoAproveitamento(Request $request)
    {
        $user = auth()->user();
        $aluno = $user->aluno;
        $anoLetivoAtivo = AnoLetivo::getAnoLetivoAtual();

        if (!$aluno) {
            return redirect()->route('aluno.dashboard')
                ->withErrors(['error' => 'Seu perfil de estudante não está completo.']);
        }

        if (!$anoLetivoAtivo) {
            return redirect()->route('aluno.dashboard')
                ->withErrors(['error' => 'Nenhum ano letivo ativo encontrado.']);
        }

        // Buscar turma atual do aluno
        $turma = $aluno->turmas()
            ->wherePivot('ano_letivo_id', $anoLetivoAtivo->id)
            ->first();

        if (!$turma) {
            return redirect()->route('aluno.dashboard')
                ->withErrors(['error' => 'Você não está matriculado em nenhuma turma no ano letivo ativo.']);
        }

        // Buscar trimestre selecionado (default: 1º trimestre)
        $trimestreNumero = $request->get('trimestre', 1);
        $trimestre = \App\Models\Trimestre::where('ano_letivo_id', $anoLetivoAtivo->id)
            ->where('numero', $trimestreNumero)
            ->first();

        if (!$trimestre) {
            return redirect()->route('aluno.dashboard')
                ->withErrors(['error' => 'Trimestre não encontrado.']);
        }

        // Buscar disciplinas da turma
        $disciplinasIds = DB::table('turma_disciplina_professor')
            ->where('turma_id', $turma->id)
            ->where('ano_letivo_id', $anoLetivoAtivo->id)
            ->pluck('disciplina_id')
            ->unique()
            ->toArray();

        // Se não encontrar disciplinas na turma_disciplina_professor, buscar das avaliações do aluno
        if (empty($disciplinasIds)) {
            $disciplinasIds = Avaliacao::where('aluno_id', $aluno->id)
                ->where('turma_id', $turma->id)
                ->where('ano_letivo_id', $anoLetivoAtivo->id)
                ->distinct()
                ->pluck('disciplina_id')
                ->toArray();
        }

        $disciplinas = \App\Models\Disciplina::whereIn('id', $disciplinasIds)
            ->orderBy('nome')
            ->get();

        // Buscar avaliações do trimestre selecionado
        $avaliacao = Avaliacao::where('aluno_id', $aluno->id)
            ->where('turma_id', $turma->id)
            ->where('trimestre_id', $trimestre->id)
            ->where('ano_letivo_id', $anoLetivoAtivo->id)
            ->first();

        // Buscar médias por disciplina
        $mediasPorDisciplina = [];
        $totalMedias = 0;
        $disciplinasComNota = 0;

        foreach ($disciplinas as $disciplina) {
            $avaliacaoDisciplina = Avaliacao::where('aluno_id', $aluno->id)
                ->where('disciplina_id', $disciplina->id)
                ->where('turma_id', $turma->id)
                ->where('trimestre_id', $trimestre->id)
                ->where('ano_letivo_id', $anoLetivoAtivo->id)
                ->first();

            $media = null;
            if ($avaliacaoDisciplina && $avaliacaoDisciplina->mt !== null) {
                $media = round($avaliacaoDisciplina->mt, 0);
                $totalMedias += $media;
                $disciplinasComNota++;
            }

            $mediasPorDisciplina[$disciplina->id] = $media;
        }

        // Calcular média trimestral geral
        $mediaTrimestralGeral = $disciplinasComNota > 0 ? round($totalMedias / $disciplinasComNota, 0) : null;

        // Classificar média trimestral
        $classificacaoMedia = $this->classificarMedia($mediaTrimestralGeral);

        // Buscar número do aluno na turma (ordem alfabética)
        $alunosTurma = $turma->alunos()
            ->wherePivot('ano_letivo_id', $anoLetivoAtivo->id)
            ->join('users', 'alunos.user_id', '=', 'users.id')
            ->orderBy('users.name')
            ->select('alunos.*')
            ->get();
        
        $numeroAluno = $alunosTurma->search(function($a) use ($aluno) {
            return $a->id === $aluno->id;
        });
        $numeroAluno = $numeroAluno !== false ? $numeroAluno + 1 : null;

        $configuracao = [
            'escola' => \App\Models\Configuracao::get('escola', 'ESCOLA PRIMARIA E COMPLETA SGE'),
        ];

        // Preparar logotipo em base64
        $logotipo = \App\Models\Configuracao::get('logotipo_escola', null);
        $logotipoBase64 = null;
        if ($logotipo && \Illuminate\Support\Facades\Storage::disk('public')->exists($logotipo)) {
            $logotipoPath = \Illuminate\Support\Facades\Storage::disk('public')->path($logotipo);
            $logotipoData = file_get_contents($logotipoPath);
            $mime = mime_content_type($logotipoPath);
            $logotipoBase64 = "data:{$mime};base64," . base64_encode($logotipoData);
        }

        return view('aluno.folheto-aproveitamento', compact(
            'aluno', 'turma', 'anoLetivoAtivo', 'disciplinas', 'trimestre',
            'mediasPorDisciplina', 'mediaTrimestralGeral', 'classificacaoMedia', 'numeroAluno', 'configuracao', 'logotipoBase64'
        ));
    }

    /**
     * Visualizar horários do estudante
     */
    public function horarios()
    {
        $user = auth()->user();
        $aluno = $user->aluno;
        $anoLetivoAtivo = AnoLetivo::getAnoLetivoAtual();

        if (!$aluno) {
            return redirect()->route('aluno.dashboard')
                ->withErrors(['error' => 'Seu perfil de estudante não está completo.']);
        }

        // Buscar turmas do estudante
        $turmas = collect();
        if ($anoLetivoAtivo) {
            $turmas = $aluno->turmas()
                ->wherePivot('ano_letivo_id', $anoLetivoAtivo->id)
                ->get();
        }

        // Buscar horários das turmas do estudante
        $horarios = collect();
        if ($anoLetivoAtivo && $turmas->isNotEmpty()) {
            $turmaIds = $turmas->pluck('id');
            $horarios = Horario::where('ano_letivo_id', $anoLetivoAtivo->id)
                ->whereIn('turma_id', $turmaIds)
                ->with(['turma', 'disciplina', 'professor.user'])
                ->orderBy('dia_semana')
                ->orderBy('hora_inicio')
                ->get();
        }

        // Organizar por turma (similar ao admin)
        $diasSemana = ['Segunda', 'Terça', 'Quarta', 'Quinta', 'Sexta'];
        $horariosPorDia = [];
        $grades = [];
        
        // Organizar horários por dia
        foreach ($diasSemana as $dia) {
            $horariosPorDia[$dia] = $horarios->filter(function($h) use ($dia) {
                return isset($h->dia_semana) && $h->dia_semana === $dia;
            })->sortBy('hora_inicio')->values();
        }
        
        // Para cada turma, criar estrutura de grade (similar ao admin)
        foreach ($turmas as $turma) {
            $horariosTurma = $horarios->filter(function($h) use ($turma) {
                return $h->turma_id == $turma->id;
            });
            
            // Obter todos os horários únicos para esta turma (com início e fim)
            $horariosUnicos = $horariosTurma->map(function($h) {
                return [
                    'inicio' => \Carbon\Carbon::parse($h->hora_inicio)->format('H:i'),
                    'fim' => \Carbon\Carbon::parse($h->hora_fim)->format('H:i'),
                    'hora_inicio' => $h->hora_inicio,
                ];
            })->unique(function($item) {
                return $item['inicio'];
            })->sortBy('hora_inicio')->values();
            
            // Criar estrutura de grade e identificar intervalo
            $grade = [];
            $horariosComIntervalo = [];
            
            foreach ($horariosUnicos as $index => $horarioUnico) {
                $horaFormatada = $horarioUnico['inicio'];
                $horaCarbon = \Carbon\Carbon::parse($horarioUnico['hora_inicio']);
                $grade[$horaFormatada] = [];
                
                foreach ($diasSemana as $dia) {
                    $horario = $horariosTurma->first(function($h) use ($dia, $horaCarbon) {
                        $horaInicio = \Carbon\Carbon::parse($h->hora_inicio);
                        return $h->dia_semana === $dia && 
                               $horaInicio->format('H:i') === $horaCarbon->format('H:i');
                    });
                    
                    $grade[$horaFormatada][$dia] = $horario;
                }
                
                // Adicionar horário à lista
                $horariosComIntervalo[] = [
                    'inicio' => $horarioUnico['inicio'],
                    'fim' => $horarioUnico['fim'],
                    'display' => $horarioUnico['inicio'] . ' - ' . $horarioUnico['fim'],
                    'tipo' => 'aula',
                ];
                
                // Verificar se há intervalo após a 3ª aula (índice 2, após criar a 3ª)
                if ($index == 2 && isset($horariosUnicos[$index + 1])) {
                    $fimTerceira = \Carbon\Carbon::parse($horarioUnico['fim']);
                    $inicioQuarta = \Carbon\Carbon::parse($horariosUnicos[$index + 1]['inicio']);
                    $diferencaMinutos = $fimTerceira->diffInMinutes($inicioQuarta);
                    
                    // Se a diferença for 15 minutos, é o intervalo de lanche
                    if ($diferencaMinutos == 15) {
                        $horariosComIntervalo[] = [
                            'inicio' => $fimTerceira->format('H:i'),
                            'fim' => $inicioQuarta->format('H:i'),
                            'display' => 'Intervalo: ' . $fimTerceira->format('H:i') . ' - ' . $inicioQuarta->format('H:i'),
                            'tipo' => 'intervalo',
                        ];
                    }
                }
            }
            
            $grades[$turma->id] = [
                'turma' => $turma,
                'grade' => $grade,
                'horarios' => $horariosComIntervalo,
            ];
        }

        return view('aluno.horarios', compact('aluno', 'anoLetivoAtivo', 'horariosPorDia', 'diasSemana', 'turmas', 'grades'));
    }

    /**
     * Visualizar trabalhos/TPCs do estudante
     */
    public function trabalhos()
    {
        $user = auth()->user();
        $aluno = $user->aluno;
        $anoLetivoAtivo = AnoLetivo::getAnoLetivoAtual();

        if (!$aluno) {
            return redirect()->route('aluno.dashboard')
                ->withErrors(['error' => 'Seu perfil de estudante não está completo.']);
        }

        $trabalhos = collect();
        if ($anoLetivoAtivo) {
            $trabalhos = $aluno->trabalhos()
                ->where('ano_letivo_id', $anoLetivoAtivo->id)
                ->with(['disciplina', 'turma', 'professor.user'])
                ->orderBy('data_entrega', 'desc')
                ->get();
        }

        return view('aluno.trabalhos', compact('aluno', 'anoLetivoAtivo', 'trabalhos'));
    }

    /**
     * Visualizar arquivo de trabalho/TPC
     */
    public function visualizarTrabalho(Trabalho $trabalho)
    {
        $user = auth()->user();
        $aluno = $user->aluno;

        if (!$aluno) {
            abort(403, 'Você não tem permissão para visualizar este arquivo.');
        }

        // Verificar se o aluno está atribuído a este trabalho
        $alunoTrabalho = $trabalho->alunos()->where('aluno_id', $aluno->id)->first();
        if (!$alunoTrabalho) {
            abort(403, 'Você não está atribuído a este trabalho.');
        }

        if (!$trabalho->arquivo || !Storage::disk('public')->exists($trabalho->arquivo)) {
            abort(404, 'Arquivo não encontrado.');
        }

        $caminhoArquivo = Storage::disk('public')->path($trabalho->arquivo);
        $mimeType = Storage::disk('public')->mimeType($trabalho->arquivo);
        
        return response()->file($caminhoArquivo, [
            'Content-Type' => $mimeType,
            'Content-Disposition' => 'inline; filename="' . $trabalho->nome_arquivo . '"',
        ]);
    }

    /**
     * Visualizar eventos do estudante
     */
    public function eventos()
    {
        $user = auth()->user();
        $aluno = $user->aluno;
        $anoLetivoAtivo = AnoLetivo::getAnoLetivoAtual();

        if (!$aluno) {
            return redirect()->route('aluno.dashboard')
                ->withErrors(['error' => 'Seu perfil de estudante não está completo.']);
        }

        // Buscar turmas do estudante
        $turmas = collect();
        if ($anoLetivoAtivo) {
            $turmas = $aluno->turmas()
                ->wherePivot('ano_letivo_id', $anoLetivoAtivo->id)
                ->get();
        }

        $eventos = collect();
        if ($anoLetivoAtivo) {
            $turmaIds = $turmas->pluck('id');
            $eventos = Evento::where('ano_letivo_id', $anoLetivoAtivo->id)
                ->where(function($query) use ($turmaIds) {
                    $query->whereNull('turma_id')
                        ->orWhereIn('turma_id', $turmaIds);
                })
                ->where('cancelado', false)
                ->orderBy('data')
                ->orderBy('hora_inicio')
                ->get();
        }

        return view('aluno.eventos', compact('aluno', 'anoLetivoAtivo', 'eventos', 'turmas'));
    }

    /**
     * Visualizar situação financeira do estudante
     */
    public function financeiro(Request $request)
    {
        $user = auth()->user();
        $aluno = $user->aluno;
        
        // Para alunos, mostrar apenas 2026 e anos anteriores (não mostrar 2027)
        // Filtrar anos letivos: apenas 2026 e anteriores
        $anosLetivos = AnoLetivo::where(function($q) {
            $q->where('ano', 'like', '%2026%')
              ->orWhere('ano', '2026')
              ->orWhere('ano', 'like', '%2025%')
              ->orWhere('ano', '2025')
              ->orWhere('ano', 'like', '%2024%')
              ->orWhere('ano', '2024')
              ->orWhere('ano', 'like', '%2023%')
              ->orWhere('ano', '2023');
        })->orderBy('ano', 'desc')->get();
        
        // Sempre usar 2026 como padrão
        $anoLetivoSelecionado = AnoLetivo::getAnoLetivoAtual();
        
        // Se houver filtro na requisição, usar esse (mas validar que não seja 2027)
        $anoLetivoId = $request->get('ano_letivo_id');
        if ($anoLetivoId) {
            $anoLetivoFiltrado = AnoLetivo::find($anoLetivoId);
            // Só permitir se não for 2027
            if ($anoLetivoFiltrado) {
                $anoRaw = $anoLetivoFiltrado->getRawOriginal('ano');
                if (strpos($anoRaw, '2027') === false && $anoRaw !== '2027') {
                    $anoLetivoAtivo = $anoLetivoFiltrado;
                } else {
                    $anoLetivoAtivo = $anoLetivoSelecionado;
                }
            } else {
                $anoLetivoAtivo = $anoLetivoSelecionado;
            }
        } else {
            $anoLetivoAtivo = $anoLetivoSelecionado;
        }

        if (!$aluno) {
            return redirect()->route('aluno.dashboard')
                ->withErrors(['error' => 'Seu perfil de estudante não está completo.']);
        }

        $financeiros = collect();
        $multasPorFinanceiro = [];
        $financeiroService = app(\App\Services\FinanceiroService::class);
        
        if ($anoLetivoAtivo) {
            // Gerar mensalidades automaticamente se não existirem
            $mensalidadesExistentes = $aluno->financeiro()
                ->where('ano_letivo_id', $anoLetivoAtivo->id)
                ->where('tipo', 'Mensalidade')
                ->count();
            
            if ($mensalidadesExistentes === 0) {
                // Gerar todas as mensalidades do ano automaticamente
                $modalidade = $aluno->modalidade_pagamento ?? 'MENSAL';
                $valorMensal = $financeiroService->getValorMensalidadePorClasse($aluno, $anoLetivoAtivo->id);
                
                if ($valorMensal > 0) {
                    $ano = (int) date('Y');
                    $meses = [1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12]; // Todos os meses
                    
                    foreach ($meses as $mes) {
                        // Verificar se já existe mensalidade para este mês
                        $existe = \App\Models\Financeiro::where('aluno_id', $aluno->id)
                            ->where('ano_letivo_id', $anoLetivoAtivo->id)
                            ->where('tipo', 'Mensalidade')
                            ->whereMonth('data_vencimento', $mes)
                            ->whereYear('data_vencimento', $ano)
                            ->exists();
                        
                        if (!$existe) {
                            $dataVencimento = \Carbon\Carbon::create($ano, $mes, 5);
                            
                            \App\Models\Financeiro::create([
                                'aluno_id' => $aluno->id,
                                'ano_letivo_id' => $anoLetivoAtivo->id,
                                'tipo' => 'Mensalidade',
                                'descricao' => "Mensalidade {$mes}/{$ano}",
                                'valor' => $valorMensal,
                                'data_vencimento' => $dataVencimento,
                                'status' => 'PENDENTE',
                            ]);
                        }
                    }
                }
            }
            
            // Verificar novamente e gerar mensalidades que faltam (mesmo que já exista pelo menos uma)
            $valorMensal = $financeiroService->getValorMensalidadePorClasse($aluno, $anoLetivoAtivo->id);
            if ($valorMensal > 0) {
                $ano = (int) date('Y');
                $meses = [1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12];
                
                foreach ($meses as $mes) {
                    $existe = \App\Models\Financeiro::where('aluno_id', $aluno->id)
                        ->where('ano_letivo_id', $anoLetivoAtivo->id)
                        ->where('tipo', 'Mensalidade')
                        ->whereMonth('data_vencimento', $mes)
                        ->whereYear('data_vencimento', $ano)
                        ->exists();
                    
                    if (!$existe) {
                        $dataVencimento = \Carbon\Carbon::create($ano, $mes, 5);
                        
                        \App\Models\Financeiro::create([
                            'aluno_id' => $aluno->id,
                            'ano_letivo_id' => $anoLetivoAtivo->id,
                            'tipo' => 'Mensalidade',
                            'descricao' => "Mensalidade {$mes}/{$ano}",
                            'valor' => $valorMensal,
                            'data_vencimento' => $dataVencimento,
                            'status' => 'PENDENTE',
                        ]);
                    }
                }
            }
            
            // Buscar todas as mensalidades (incluindo futuras) para permitir pagamento adiantado
            $financeiros = $aluno->financeiro()
                ->where('ano_letivo_id', $anoLetivoAtivo->id)
                ->where('tipo', 'Mensalidade')
                ->orderBy('data_vencimento')
                ->get();
            
            // Calcular multas para cada mensalidade
            foreach ($financeiros as $financeiro) {
                if ($financeiro->tipo === 'Mensalidade') {
                    $multa = $financeiroService->calcularMulta($financeiro);
                    $valorComMulta = $financeiroService->getValorComMulta($financeiro);
                    
                    $diasAtraso = 0;
                    if ($financeiro->data_vencimento < now()) {
                        $hoje = now()->startOfDay();
                        $vencimento = \Carbon\Carbon::parse($financeiro->data_vencimento)->startOfDay();
                        $diasAtraso = $hoje->diffInDays($vencimento);
                    }
                    
                    $multasPorFinanceiro[$financeiro->id] = [
                        'multa' => $multa,
                        'valor_com_multa' => $valorComMulta,
                        'dias_atraso' => $diasAtraso,
                    ];
                }
            }
        }

        // Usar o método do serviço para garantir consistência com período de graça
        $situacaoFinanceira = 'REGULAR';
        if ($anoLetivoAtivo) {
            $financeiroService = app(\App\Services\FinanceiroService::class);
            $situacaoFinanceira = $financeiroService->getSituacaoFinanceira($aluno, $anoLetivoAtivo->id);
        }
        
        // Considerar apenas mensalidades vencidas há mais de 4 dias para cálculos de dívida (a partir do dia 6)
        // Última data sem multa: dia 5. A partir do dia 6 começa a multa.
        $dataLimite = now()->subDays(4);
        
        $totalPago = $financeiros->where('status', 'PAGO')->sum('valor');
        
        // Total pendente: apenas mensalidades vencidas há mais de 4 dias (a partir do dia 6)
        $totalPendente = $financeiros->whereIn('status', ['PENDENTE', 'VENCIDO'])
            ->filter(function($f) use ($dataLimite) {
                return $f->data_vencimento <= $dataLimite;
            })
            ->sum('valor');
        
        // Total vencido: apenas mensalidades vencidas há mais de 4 dias (a partir do dia 6)
        $totalVencido = $financeiros->where('status', 'VENCIDO')
            ->filter(function($f) use ($dataLimite) {
                return $f->data_vencimento <= $dataLimite;
            })
            ->sum('valor');

        // Buscar facturas do aluno (filtrar por ano letivo se especificado)
        $invoicesQuery = $aluno->invoices();
        if ($anoLetivoAtivo) {
            $invoicesQuery->where('ano_letivo_id', $anoLetivoAtivo->id);
        }
        $invoices = $invoicesQuery->with('payment')
            ->orderBy('created_at', 'desc')
            ->get();

        // Calcular totais incluindo multas
        $totalMultas = 0;
        $totalComMultas = $totalPendente;
        foreach ($multasPorFinanceiro as $multaData) {
            $totalMultas += $multaData['multa'];
        }
        $totalComMultas += $totalMultas;

        // Buscar turma do aluno no ano letivo ativo
        $turma = null;
        if ($anoLetivoAtivo) {
            $turma = $aluno->turmas()
                ->wherePivot('ano_letivo_id', $anoLetivoAtivo->id)
                ->first();
        }

        return view('aluno.financeiro', compact(
            'aluno',
            'anoLetivoAtivo',
            'anosLetivos',
            'financeiros',
            'multasPorFinanceiro',
            'turma',
            'totalMultas',
            'totalComMultas',
            'situacaoFinanceira',
            'totalPago',
            'totalPendente',
            'totalVencido',
            'invoices'
        ));
    }

    /**
     * Visualizar perfil do estudante
     */
    public function perfil()
    {
        $user = auth()->user();
        $aluno = $user->aluno;

        if (!$aluno) {
            return redirect()->route('aluno.dashboard')
                ->withErrors(['error' => 'Seu perfil de estudante não está completo.']);
        }

        $aluno->load('user');

        // Buscar turmas atuais
        $anoLetivoAtivo = AnoLetivo::getAnoLetivoAtual();
        $turmas = collect();
        if ($anoLetivoAtivo) {
            $turmas = $aluno->turmas()
                ->wherePivot('ano_letivo_id', $anoLetivoAtivo->id)
                ->get();
        }

        return view('aluno.perfil', compact('aluno', 'turmas', 'anoLetivoAtivo'));
    }
}
