<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\Auditoria;
use App\Models\LoginAttempt;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class AtividadeController extends Controller
{
    public function __construct()
    {
        $this->middleware(['auth', 'role:admin,superadmin,funcionario']);
    }

    /**
     * Verifica se o usuário é admin ou superadmin (apenas eles veem atividade)
     */
    private function podeAcessar(): bool
    {
        $user = auth()->user();
        return $user->isAdmin() || $user->isSuperadmin();
    }

    public function index(Request $request)
    {
        if (!$this->podeAcessar()) {
            abort(403, 'Acesso negado. Apenas admin e superadmin podem ver esta página.');
        }

        $limite = (int) ($request->input('limite', 50));
        $limite = min(max($limite, 10), 200);

        // Últimos logins (LoginAttempt success=true, ordenado por created_at)
        // Excluir superadmin - tornar anônimo
        $ultimosLogins = LoginAttempt::with('user')
            ->where('success', true)
            ->where(function($query) {
                $query->whereHas('user', function($q) {
                    $q->where('tipo', '!=', 'superadmin');
                })->orWhereNull('user_id');
            })
            ->orderByDesc('created_at')
            ->limit($limite)
            ->get();

        // Usuários online: sessões com last_activity nos últimos 5 minutos
        $minutosOnline = 5;
        $limiteUnix = time() - ($minutosOnline * 60);
        $sessoesAtivas = collect();
        try {
            $sessoesAtivas = DB::table('sessions')
                ->whereNotNull('user_id')
                ->where('last_activity', '>=', $limiteUnix)
                ->select('user_id', DB::raw('MAX(last_activity) as last_activity'))
                ->groupBy('user_id')
                ->get();
        } catch (\Exception $e) {
            // tabela sessions pode não existir (driver file)
        }

        $userIds = $sessoesAtivas->pluck('user_id')->unique()->filter()->values()->all();
        $usersOnline = $userIds ? User::whereIn('id', $userIds)
            ->where('tipo', '!=', 'superadmin')
            ->get()->keyBy('id') : collect();
        $onlineComAtividade = $sessoesAtivas->map(function ($s) use ($usersOnline) {
            $u = $usersOnline->get($s->user_id);
            return $u ? ['user' => $u, 'last_activity' => $s->last_activity] : null;
        })->filter()->values();

        // Tentativas de login (falhas)
        // Excluir superadmin - tornar anônimo
        $tentativasFalhas = LoginAttempt::with('user')
            ->where('success', false)
            ->where(function($query) {
                $query->whereHas('user', function($q) {
                    $q->where('tipo', '!=', 'superadmin');
                })->orWhereNull('user_id');
            })
            ->orderByDesc('created_at')
            ->limit($limite)
            ->get();

        return view('admin.atividade-logins', compact(
            'ultimosLogins',
            'onlineComAtividade',
            'tentativasFalhas',
            'limite',
            'minutosOnline'
        ));
    }
}
