<?php

namespace App\Http\Controllers;

use App\Models\Message;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class ChatController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Lista conversas do usuário
     */
    public function index()
    {
        $user = Auth::user();
        
        // Buscar conversas (usuários com quem já trocou mensagens)
        // Superadmin pode ver todas as conversas, mas não aparece como destinatário
        if ($user->isSuperadmin()) {
            // Superadmin vê todas as conversas do sistema
            $conversations = Message::with(['sender', 'receiver'])
                ->orderBy('created_at', 'desc')
                ->get()
                ->groupBy(function($message) {
                    // Agrupar por par de usuários (ordenado para consistência)
                    $ids = [$message->sender_id, $message->receiver_id];
                    sort($ids);
                    return implode('-', $ids);
                })
                ->map(function($messages, $key) use ($user) {
                    $firstMessage = $messages->first();
                    $otherUser = $firstMessage->sender_id == $user->id 
                        ? $firstMessage->receiver 
                        : $firstMessage->sender;
                    
                    // Se a conversa é com superadmin, mostrar o outro usuário
                    if ($firstMessage->sender_id == $user->id) {
                        $otherUser = $firstMessage->receiver;
                    } elseif ($firstMessage->receiver_id == $user->id) {
                        $otherUser = $firstMessage->sender;
                    } else {
                        // Superadmin vendo conversa entre outros usuários
                        $otherUser = $firstMessage->sender->tipo != 'superadmin' 
                            ? $firstMessage->sender 
                            : $firstMessage->receiver;
                    }
                    
                    if (!$otherUser || $otherUser->tipo === 'superadmin') {
                        return null;
                    }
                    
                    $lastMessage = $messages->first();
                    $unreadCount = 0; // Superadmin não tem mensagens não lidas (ele só observa)
                    
                    return [
                        'user' => $otherUser,
                        'last_message' => $lastMessage,
                        'unread_count' => $unreadCount,
                    ];
                })
                ->filter()
                ->sortByDesc(function($conversation) {
                    return $conversation['last_message']->created_at;
                });
        } else {
            // Usuários normais veem apenas suas conversas (otimizado)
            // Buscar apenas últimas mensagens por conversa (limitado a 100 conversas)
            $conversations = Message::select('id', 'sender_id', 'receiver_id', 'message', 'read', 'read_at', 'created_at', 'updated_at')
                ->where(function($query) use ($user) {
                    $query->where('sender_id', $user->id)
                          ->orWhere('receiver_id', $user->id);
                })
                ->with(['sender:id,name,email', 'receiver:id,name,email'])
                ->orderBy('created_at', 'desc')
                ->limit(1000) // Limitar para performance
                ->get()
                ->groupBy(function($message) use ($user) {
                    return $message->sender_id == $user->id ? $message->receiver_id : $message->sender_id;
                })
                ->map(function($messages, $userId) use ($user) {
                    $otherUser = $messages->first()->sender_id == $user->id 
                        ? $messages->first()->receiver 
                        : $messages->first()->sender;
                    
                    // Filtrar superadmin das conversas
                    if (!$otherUser || $otherUser->tipo === 'superadmin') {
                        return null;
                    }
                    
                    $lastMessage = $messages->first();
                    // Contar apenas mensagens não lidas desta conversa
                    $unreadCount = Message::where(function($q) use ($user, $userId) {
                            $q->where('sender_id', $userId)
                              ->where('receiver_id', $user->id);
                        })
                        ->where('read', false)
                        ->count();
                    
                    return [
                        'user' => $otherUser,
                        'last_message' => $lastMessage,
                        'unread_count' => $unreadCount,
                    ];
                })
                ->filter()
                ->sortByDesc(function($conversation) {
                    return $conversation['last_message']->created_at;
                })
                ->take(50); // Limitar a 50 conversas exibidas
        }

        // Buscar usuários disponíveis para iniciar conversa
        $availableUsers = collect();
        if ($user->isAdmin() || $user->isSuperadmin()) {
            // Admin pode ver professores e alunos, mas superadmin não aparece nas listas
            $availableUsers = User::where('id', '!=', $user->id)
                ->whereIn('tipo', ['professor', 'aluno'])
                ->where('tipo', '!=', 'superadmin') // Garantir que superadmin nunca aparece
                ->with(['professor', 'aluno'])
                ->get();
        } elseif ($user->isAluno()) {
            // Alunos podem conversar com professores que lhes dão aulas e admin (direção)
            $aluno = $user->aluno;
            $anoLetivoAtivo = \App\Models\AnoLetivo::where('ativo', true)->first();
            
            $professoresIds = collect();
            
            if ($aluno && $anoLetivoAtivo) {
                // Buscar turmas do aluno no ano letivo ativo
                $turmasAluno = $aluno->turmas()
                    ->wherePivot('ano_letivo_id', $anoLetivoAtivo->id)
                    ->pluck('turmas.id');
                
                if ($turmasAluno->isNotEmpty()) {
                    // Buscar professores que lecionam nessas turmas
                    $professoresIds = \DB::table('turma_disciplina_professor')
                        ->whereIn('turma_id', $turmasAluno)
                        ->where('ano_letivo_id', $anoLetivoAtivo->id)
                        ->distinct()
                        ->pluck('professor_id');
                }
            }
            
            // Buscar usuários dos professores que lecionam ao aluno + admins (direção)
            $availableUsers = User::where('id', '!=', $user->id)
                ->where(function($q) use ($professoresIds) {
                    // Professores que lecionam ao aluno
                    if ($professoresIds->isNotEmpty()) {
                        $q->where(function($subQ) use ($professoresIds) {
                            $subQ->where('tipo', 'professor')
                                 ->whereHas('professor', function($profQ) use ($professoresIds) {
                                     $profQ->whereIn('id', $professoresIds);
                                 });
                        });
                    }
                    // Ou admins (direção)
                    $q->orWhere('tipo', 'admin');
                })
                ->where('tipo', '!=', 'superadmin') // Garantir que superadmin nunca aparece
                ->with(['professor'])
                ->get();
        } elseif ($user->isProfessor()) {
            // Professores podem conversar com alunos
            $availableUsers = User::where('id', '!=', $user->id)
                ->where('tipo', 'aluno')
                ->where('tipo', '!=', 'superadmin') // Garantir que superadmin nunca aparece
                ->with(['aluno'])
                ->get();
        }

        return view('chat.index', compact('conversations', 'availableUsers'));
    }

    /**
     * Mostra conversa específica com um usuário
     */
    public function show(User $user)
    {
        $currentUser = Auth::user();
        
        // Superadmin não pode aparecer como destinatário
        if ($user->tipo === 'superadmin' && $currentUser->tipo !== 'superadmin') {
            abort(404, 'Usuário não encontrado.');
        }
        
        // Se superadmin está visualizando, pode ver qualquer conversa
        if ($currentUser->isSuperadmin()) {
            // Superadmin pode ver conversas entre outros usuários
            // Buscar todas as mensagens envolvendo este usuário
            $messages = Message::where(function($query) use ($user) {
                    $query->where('sender_id', $user->id)
                          ->orWhere('receiver_id', $user->id);
                })
                ->with(['sender', 'receiver'])
                ->orderBy('created_at', 'asc')
                ->get();
        } else {
            // Verificar permissões de chat baseado nos tipos de usuário
            if (!$this->canChat($currentUser, $user)) {
                abort(403, 'Você não tem permissão para conversar com este usuário.');
            }

            // Buscar mensagens entre os dois usuários (otimizado)
            $messages = Message::where(function($query) use ($currentUser, $user) {
                    $query->where('sender_id', $currentUser->id)
                          ->where('receiver_id', $user->id);
                })
                ->orWhere(function($query) use ($currentUser, $user) {
                    $query->where('sender_id', $user->id)
                          ->where('receiver_id', $currentUser->id);
                })
                ->select('id', 'sender_id', 'receiver_id', 'message', 'read', 'read_at', 'file_path', 'file_name', 'file_type', 'file_size', 'created_at', 'updated_at')
                ->with(['sender:id,name,email', 'receiver:id,name,email'])
                ->orderBy('created_at', 'asc')
                ->get();

            // Marcar mensagens como lidas (apenas se não for superadmin)
            $updated = Message::where('sender_id', $user->id)
                ->where('receiver_id', $currentUser->id)
                ->where('read', false)
                ->update(['read' => true, 'read_at' => now()]);
            
            // Invalidar cache se houve atualização
            if ($updated > 0) {
                \Cache::forget("unread_messages_count_{$currentUser->id}");
            }
        }

        return view('chat.show', compact('user', 'messages'));
    }

    /**
     * Enviar mensagem
     */
    public function store(Request $request)
    {
        $request->validate([
            'receiver_id' => 'required|exists:users,id',
            'message' => 'nullable|string|max:5000',
            'file' => 'nullable|file|max:10240|mimes:pdf,doc,docx,xls,xlsx,ppt,pptx,txt,jpg,jpeg,png,gif,zip,rar',
        ]);

        $sender = Auth::user();
        $receiver = User::findOrFail($request->receiver_id);

        // Superadmin não pode receber mensagens de outros usuários
        if ($receiver->tipo === 'superadmin' && $sender->tipo !== 'superadmin') {
            return response()->json(['error' => 'Não é possível enviar mensagem para este usuário.'], 403);
        }

        // Superadmin não pode enviar mensagens (só observa)
        if ($sender->isSuperadmin()) {
            return response()->json(['error' => 'Superadmin não pode enviar mensagens. Apenas visualiza.'], 403);
        }

        // Verificar permissões
        if (!$this->canChat($sender, $receiver)) {
            return response()->json(['error' => 'Você não tem permissão para conversar com este usuário.'], 403);
        }

        // Validar que há mensagem ou arquivo
        if (empty($request->message) && !$request->hasFile('file')) {
            return response()->json(['error' => 'É necessário enviar uma mensagem ou um arquivo.'], 422);
        }

        $messageData = [
            'sender_id' => $sender->id,
            'receiver_id' => $receiver->id,
            'message' => $request->message ?? '',
        ];

        // Processar upload de arquivo
        if ($request->hasFile('file')) {
            $arquivo = $request->file('file');
            $nomeArquivo = time() . '_' . uniqid() . '.' . $arquivo->getClientOriginalExtension();
            $caminho = $arquivo->storeAs('chat/files', $nomeArquivo, 'public');
            
            $messageData['file_path'] = $caminho;
            $messageData['file_name'] = $arquivo->getClientOriginalName();
            $messageData['file_type'] = $arquivo->getMimeType();
            $messageData['file_size'] = $arquivo->getSize();
        }

        $message = Message::create($messageData);

        // Invalidar cache de contador de mensagens não lidas
        \Cache::forget("unread_messages_count_{$receiver->id}");

        // Carregar relacionamentos necessários
        $message->load(['sender', 'receiver']);

        return response()->json([
            'success' => true,
            'message' => [
                'id' => $message->id,
                'sender_id' => $message->sender_id,
                'receiver_id' => $message->receiver_id,
                'message' => $message->message,
                'file_path' => $message->file_path,
                'file_name' => $message->file_name,
                'file_type' => $message->file_type,
                'file_size' => $message->file_size,
                'created_at' => $message->created_at->toISOString(),
                'updated_at' => $message->updated_at->toISOString(),
                'sender' => [
                    'id' => $message->sender->id,
                    'name' => $message->sender->name,
                    'email' => $message->sender->email,
                ],
            ],
        ], 200);
    }

    /**
     * Criar nova mensagem (para admin criar mensagens em massa)
     */
    public function create(Request $request)
    {
        $user = Auth::user();
        $tipo = $request->get('tipo');
        
        // Superadmin não pode enviar mensagens
        if ($user->isSuperadmin()) {
            abort(403, 'Superadmin não pode enviar mensagens. Apenas visualiza.');
        }
        
        if (!$user->isAdmin() && !$user->isAluno()) {
            abort(403);
        }

        return view('chat.create', compact('tipo'));
    }

    /**
     * Enviar mensagem em massa (admin)
     */
    public function storeMass(Request $request)
    {
        $request->validate([
            'tipo' => 'required|in:geral,professores,alunos,escola',
            'message' => 'required|string|max:5000',
        ]);

        $sender = Auth::user();
        $tipo = $request->tipo;
        $message = $request->message;
        $enviadas = 0;

        if ($sender->isAdmin() || $sender->isSuperadmin()) {
            if ($tipo === 'geral') {
                // Enviar para todos os usuários (exceto superadmin)
                $receivers = User::where('id', '!=', $sender->id)
                    ->where('tipo', '!=', 'superadmin')
                    ->get();
            } elseif ($tipo === 'professores') {
                // Enviar para todos os professores
                $receivers = User::where('tipo', 'professor')->get();
            } elseif ($tipo === 'alunos') {
                // Enviar para todos os alunos
                $receivers = User::where('tipo', 'aluno')->get();
            } else {
                return response()->json(['error' => 'Tipo inválido'], 400);
            }

            foreach ($receivers as $receiver) {
                // Nunca enviar para superadmin
                if ($receiver->tipo === 'superadmin') {
                    continue;
                }
                
                Message::create([
                    'sender_id' => $sender->id,
                    'receiver_id' => $receiver->id,
                    'message' => $message,
                    'conversation_type' => 'group',
                ]);
                $enviadas++;
            }
        } elseif ($sender->isAluno()) {
            if ($tipo === 'escola' || $tipo === 'direcao') {
                // Enviar apenas para admins (não superadmin) - direção
                $receivers = User::where('tipo', 'admin')
                    ->where('tipo', '!=', 'superadmin')
                    ->get();
                
                foreach ($receivers as $receiver) {
                    Message::create([
                        'sender_id' => $sender->id,
                        'receiver_id' => $receiver->id,
                        'message' => $message,
                        'conversation_type' => 'direct',
                    ]);
                    $enviadas++;
                }
            } elseif ($tipo === 'professor') {
                // Enviar para professor específico (validado pelo canChat)
                $request->validate([
                    'user_id' => 'required|exists:users,id',
                ]);
                
                $receiver = User::findOrFail($request->user_id);
                
                // Verificar se pode conversar (canChat já valida se é professor que leciona ao aluno)
                if (!$this->canChat($sender, $receiver)) {
                    return response()->json(['error' => 'Você não tem permissão para conversar com este usuário.'], 403);
                }
                
                Message::create([
                    'sender_id' => $sender->id,
                    'receiver_id' => $receiver->id,
                    'message' => $message,
                    'conversation_type' => 'direct',
                ]);
                $enviadas++;
            } else {
                return response()->json(['error' => 'Tipo inválido'], 400);
            }
        }

        return response()->json([
            'success' => true,
            'message' => "Mensagem enviada para {$enviadas} destinatário(s).",
            'enviadas' => $enviadas,
        ]);
    }

    /**
     * Verificar se dois usuários podem conversar
     */
    private function canChat(User $user1, User $user2)
    {
        // Superadmin nunca pode ser destinatário, mas pode ver conversas
        if ($user2->tipo === 'superadmin' && $user1->tipo !== 'superadmin') {
            return false; // Outros usuários não podem iniciar conversa com superadmin
        }
        if ($user1->tipo === 'superadmin' && $user2->tipo === 'superadmin') {
            return false; // Superadmin não conversa com superadmin
        }
        
        // Superadmin pode ver qualquer conversa (mas não aparece como opção)
        if ($user1->isSuperadmin()) {
            return true;
        }

        // Admin pode conversar com qualquer um (exceto superadmin)
        if ($user1->isAdmin() && $user2->tipo !== 'superadmin') {
            return true;
        }
        if ($user2->isAdmin() && $user1->tipo !== 'superadmin') {
            return true;
        }

        // Professor pode conversar com alunos
        if ($user1->isProfessor() && $user2->isAluno()) {
            return true;
        }
        if ($user1->isAluno() && $user2->isProfessor()) {
            return true;
        }

        // Aluno pode conversar com admin e professores (não superadmin)
        if ($user1->isAluno() && ($user2->isAdmin() || $user2->isProfessor())) {
            return true;
        }
        if ($user2->isAluno() && ($user1->isAdmin() || $user1->isProfessor())) {
            return true;
        }

        // Mesmo tipo pode conversar (professor com professor, etc) - mas não superadmin
        if ($user1->tipo === $user2->tipo && $user1->tipo !== 'superadmin') {
            return true;
        }

        return false;
    }

    /**
     * Contar mensagens não lidas
     */
    public function unreadCount()
    {
        $user = Auth::user();
        
        // Usar cache de 5 segundos para reduzir queries
        $cacheKey = "unread_messages_count_{$user->id}";
        $count = \Cache::remember($cacheKey, 5, function() use ($user) {
            return Message::where('receiver_id', $user->id)
                ->where('read', false)
                ->count();
        });
        
        return response()->json(['count' => $count]);
    }

    /**
     * Indicador de digitação
     */
    public function typing(Request $request)
    {
        $request->validate([
            'receiver_id' => 'required|exists:users,id',
            'is_typing' => 'required|boolean',
        ]);

        $sender = Auth::user();
        $receiverId = $request->receiver_id;
        
        // Armazenar estado de digitação (usando cache com TTL curto)
        $key = "typing:{$sender->id}:{$receiverId}";
        if ($request->is_typing) {
            \Cache::put($key, true, 3); // 3 segundos
        } else {
            \Cache::forget($key);
        }
        
        return response()->json(['success' => true]);
    }

    /**
     * Verificar se alguém está digitando
     */
    public function checkTyping(Request $request)
    {
        $request->validate([
            'sender_id' => 'required|exists:users,id',
        ]);

        $receiver = Auth::user();
        $senderId = $request->sender_id;
        
        $key = "typing:{$senderId}:{$receiver->id}";
        $isTyping = \Cache::has($key);
        
        return response()->json(['is_typing' => $isTyping]);
    }

    /**
     * Marcar mensagens como lidas
     */
    public function markAsRead(Request $request)
    {
        $request->validate([
            'user_id' => 'required|exists:users,id',
        ]);

        $currentUser = Auth::user();
        $otherUserId = $request->user_id;

        // Marcar mensagens recebidas como lidas
        $updated = Message::where('sender_id', $otherUserId)
            ->where('receiver_id', $currentUser->id)
            ->where('read', false)
            ->update(['read' => true, 'read_at' => now()]);

        // Invalidar cache
        if ($updated > 0) {
            \Cache::forget("unread_messages_count_{$currentUser->id}");
        }

        return response()->json([
            'success' => true,
            'updated' => $updated,
        ]);
    }

    /**
     * Buscar novas mensagens desde uma data específica
     */
    public function getNewMessages(Request $request)
    {
        $request->validate([
            'user_id' => 'required|exists:users,id',
            'last_message_id' => 'nullable|integer',
        ]);

        $currentUser = Auth::user();
        $otherUserId = $request->user_id;
        
        $query = Message::where(function($q) use ($currentUser, $otherUserId) {
                $q->where(function($q2) use ($currentUser, $otherUserId) {
                    $q2->where('sender_id', $currentUser->id)
                       ->where('receiver_id', $otherUserId);
                })
                ->orWhere(function($q2) use ($currentUser, $otherUserId) {
                    $q2->where('sender_id', $otherUserId)
                       ->where('receiver_id', $currentUser->id);
                });
            })
            ->select('id', 'sender_id', 'receiver_id', 'message', 'read', 'read_at', 'file_path', 'file_name', 'file_type', 'file_size', 'created_at', 'updated_at')
            ->with(['sender:id,name,email', 'receiver:id,name,email']);
        
        if ($request->last_message_id) {
            $query->where('id', '>', $request->last_message_id);
        }
        
        // Limitar a 50 mensagens por vez para performance
        $messages = $query->orderBy('created_at', 'asc')->limit(50)->get();
        
        // Marcar mensagens recebidas como lidas
        $updated = Message::where('sender_id', $otherUserId)
            ->where('receiver_id', $currentUser->id)
            ->where('read', false)
            ->update(['read' => true, 'read_at' => now()]);
        
        // Invalidar cache se houve atualização
        if ($updated > 0) {
            \Cache::forget("unread_messages_count_{$currentUser->id}");
        }
        
        return response()->json([
            'messages' => $messages,
            'count' => $messages->count(),
        ]);
    }

    /**
     * Deletar mensagem
     */
    public function destroy($messageId)
    {
        try {
            $user = Auth::user();
            
            // Buscar mensagem (incluindo soft deleted para verificar se existe)
            $message = Message::withTrashed()->find($messageId);
            
            if (!$message) {
                return response()->json(['error' => 'Mensagem não encontrada.'], 404);
            }

            // Verificar se o usuário é o remetente ou destinatário
            if ($message->sender_id !== $user->id && $message->receiver_id !== $user->id) {
                return response()->json(['error' => 'Você não tem permissão para deletar esta mensagem.'], 403);
            }

            // Deletar arquivo se existir
            if ($message->file_path && Storage::disk('public')->exists($message->file_path)) {
                try {
                    Storage::disk('public')->delete($message->file_path);
                } catch (\Exception $e) {
                    // Continuar mesmo se não conseguir deletar o arquivo
                    \Log::warning('Erro ao deletar arquivo da mensagem: ' . $e->getMessage());
                }
            }

            // Deletar mensagem (soft delete)
            $message->delete();

            // Invalidar cache
            \Cache::forget("unread_messages_count_{$user->id}");

            return response()->json([
                'success' => true, 
                'message' => 'Mensagem deletada com sucesso.'
            ]);
        } catch (\Exception $e) {
            \Log::error('Erro ao deletar mensagem: ' . $e->getMessage());
            return response()->json([
                'error' => 'Erro ao deletar mensagem. Tente novamente.',
                'success' => false
            ], 500);
        }
    }

    /**
     * Deletar toda a conversa
     */
    public function deleteConversation(Request $request)
    {
        $request->validate([
            'user_id' => 'required|exists:users,id',
        ]);

        $currentUser = Auth::user();
        $otherUserId = $request->user_id;

        // Buscar todas as mensagens da conversa
        $messages = Message::where(function($query) use ($currentUser, $otherUserId) {
                $query->where('sender_id', $currentUser->id)
                      ->where('receiver_id', $otherUserId);
            })
            ->orWhere(function($query) use ($currentUser, $otherUserId) {
                $query->where('sender_id', $otherUserId)
                      ->where('receiver_id', $currentUser->id);
            })
            ->get();

        // Deletar arquivos associados
        foreach ($messages as $message) {
            if ($message->file_path && Storage::disk('public')->exists($message->file_path)) {
                Storage::disk('public')->delete($message->file_path);
            }
        }

        // Deletar mensagens
        $deleted = Message::where(function($query) use ($currentUser, $otherUserId) {
                $query->where('sender_id', $currentUser->id)
                      ->where('receiver_id', $otherUserId);
            })
            ->orWhere(function($query) use ($currentUser, $otherUserId) {
                $query->where('sender_id', $otherUserId)
                      ->where('receiver_id', $currentUser->id);
            })
            ->delete();

        // Invalidar cache
        \Cache::forget("unread_messages_count_{$currentUser->id}");

        return response()->json([
            'success' => true,
            'message' => 'Conversa deletada com sucesso.',
            'deleted' => $deleted,
        ]);
    }
}
