<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\Funcionario;
use App\Models\Configuracao;
use App\Repositories\UserRepository;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class FuncionarioController extends Controller
{
    protected $userRepository;

    public function __construct(UserRepository $userRepository)
    {
        $this->middleware(['auth', 'role:admin,superadmin']);
        $this->userRepository = $userRepository;
    }

    /**
     * Listar todos os funcionários
     */
    public function index(Request $request)
    {
        $query = Funcionario::with('user');

        if ($request->has('search') && $request->search) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->whereHas('user', function($subQ) use ($search) {
                    $subQ->where('name', 'like', "%{$search}%")
                         ->orWhere('email', 'like', "%{$search}%");
                })->orWhere('numero_funcionario', 'like', "%{$search}%")
                  ->orWhere('cargo', 'like', "%{$search}%");
            });
        }

        // Otimização: Paginação e select específico (incluindo numero_funcionario e campos de permissões)
        $funcionarios = $query->select(
                'funcionarios.id', 
                'funcionarios.user_id', 
                'funcionarios.numero_funcionario',
                'funcionarios.cargo', 
                'funcionarios.gerir_alunos',
                'funcionarios.gerir_professores',
                'funcionarios.gerir_turmas',
                'funcionarios.gerir_disciplinas',
                'funcionarios.gerir_mensalidades',
                'funcionarios.gerir_horarios',
                'funcionarios.gerir_eventos',
                'funcionarios.gerir_planos',
                'funcionarios.gerir_avaliacoes',
                'funcionarios.gerir_inscricoes',
                'funcionarios.gerir_recuperacao_senha',
                'funcionarios.gerir_pagamentos_professores',
                'funcionarios.created_at'
            )
            ->with('user:id,name,email,is_active')
            ->orderBy('created_at', 'desc')
            ->paginate(25)
            ->onEachSide(2)
            ->appends(request()->query());

        return view('admin.funcionarios.index', compact('funcionarios'));
    }

    /**
     * Mostrar formulário de criação
     */
    public function create()
    {
        return view('admin.funcionarios.create');
    }

    /**
     * Gerar número de funcionário automaticamente
     */
    private function gerarNumeroFuncionario()
    {
        $anoAtual = date('Y');
        $prefixo = "FUNC{$anoAtual}";
        
        // Busca o maior número existente ordenando numericamente
        $ultimoFuncionario = Funcionario::where('numero_funcionario', 'like', "{$prefixo}%")
            ->orderByRaw("CAST(SUBSTRING(numero_funcionario, " . (strlen($prefixo) + 1) . ") AS UNSIGNED) DESC")
            ->first();

        if ($ultimoFuncionario && $ultimoFuncionario->numero_funcionario) {
            $ultimoNumero = (int) substr($ultimoFuncionario->numero_funcionario, -5);
            $novoNumero = $ultimoNumero + 1;
        } else {
            $novoNumero = 1;
        }

        $novoCodigo = $prefixo . str_pad($novoNumero, 5, '0', STR_PAD_LEFT);
        
        // Verificar se o código já existe (segurança extra)
        while (Funcionario::where('numero_funcionario', $novoCodigo)->exists()) {
            $novoNumero++;
            $novoCodigo = $prefixo . str_pad($novoNumero, 5, '0', STR_PAD_LEFT);
        }
        
        return $novoCodigo;
    }

    /**
     * Gerar email automaticamente no formato primeiraLetraSobrenome@nomedocolegio.edu.mz
     * Exemplo: "Alcidio Fondo" -> "afondo@nomedocolegio.edu.mz"
     */
    private function gerarEmail($nome)
    {
        // Obter nome do colégio das configurações
        $nomeColegio = Configuracao::get('escola', 'colegio');
        
        // Normalizar nome do colégio: remover acentos, espaços, caracteres especiais
        $nomeColegioNormalizado = mb_strtolower($nomeColegio, 'UTF-8');
        $nomeColegioNormalizado = iconv('UTF-8', 'ASCII//TRANSLIT//IGNORE', $nomeColegioNormalizado);
        if ($nomeColegioNormalizado === false) {
            $nomeColegioNormalizado = mb_strtolower($nomeColegio, 'UTF-8');
        }
        $nomeColegioNormalizado = preg_replace('/[^a-z0-9]/', '', $nomeColegioNormalizado);
        if (empty($nomeColegioNormalizado)) {
            $nomeColegioNormalizado = 'colegio';
        }
        
        // Processar nome do funcionário: primeira letra do primeiro nome + sobrenome completo
        $partesNome = preg_split('/\s+/', trim($nome));
        
        if (count($partesNome) >= 2) {
            // Tem primeiro nome e sobrenome
            $primeiraLetra = mb_substr($partesNome[0], 0, 1);
            $sobrenome = end($partesNome); // Pega o último elemento (sobrenome)
            
            // Normalizar primeira letra
            $primeiraLetra = mb_strtolower($primeiraLetra, 'UTF-8');
            $primeiraLetra = iconv('UTF-8', 'ASCII//TRANSLIT//IGNORE', $primeiraLetra);
            if ($primeiraLetra === false) {
                $primeiraLetra = mb_strtolower(mb_substr($partesNome[0], 0, 1), 'UTF-8');
            }
            $primeiraLetra = preg_replace('/[^a-z0-9]/', '', $primeiraLetra);
            
            // Normalizar sobrenome
            $sobrenomeNormalizado = mb_strtolower($sobrenome, 'UTF-8');
            $sobrenomeNormalizado = iconv('UTF-8', 'ASCII//TRANSLIT//IGNORE', $sobrenomeNormalizado);
            if ($sobrenomeNormalizado === false) {
                $sobrenomeNormalizado = mb_strtolower($sobrenome, 'UTF-8');
            }
            $sobrenomeNormalizado = preg_replace('/[^a-z0-9]/', '', $sobrenomeNormalizado);
            
            $nomeNormalizado = $primeiraLetra . $sobrenomeNormalizado;
        } else {
            // Apenas um nome, usar primeira letra + resto do nome
            $nomeCompleto = $partesNome[0];
            $primeiraLetra = mb_substr($nomeCompleto, 0, 1);
            $restoNome = mb_substr($nomeCompleto, 1);
            
            // Normalizar
            $primeiraLetra = mb_strtolower($primeiraLetra, 'UTF-8');
            $primeiraLetra = iconv('UTF-8', 'ASCII//TRANSLIT//IGNORE', $primeiraLetra);
            if ($primeiraLetra === false) {
                $primeiraLetra = mb_strtolower(mb_substr($nomeCompleto, 0, 1), 'UTF-8');
            }
            $primeiraLetra = preg_replace('/[^a-z0-9]/', '', $primeiraLetra);
            
            $restoNormalizado = mb_strtolower($restoNome, 'UTF-8');
            $restoNormalizado = iconv('UTF-8', 'ASCII//TRANSLIT//IGNORE', $restoNormalizado);
            if ($restoNormalizado === false) {
                $restoNormalizado = mb_strtolower($restoNome, 'UTF-8');
            }
            $restoNormalizado = preg_replace('/[^a-z0-9]/', '', $restoNormalizado);
            
            $nomeNormalizado = $primeiraLetra . $restoNormalizado;
        }
        
        if (empty($nomeNormalizado)) {
            $nomeNormalizado = 'funcionario';
        }
        
        // Gerar email base
        $emailBase = $nomeNormalizado . '@' . $nomeColegioNormalizado . '.edu.mz';
        
        // Verificar se já existe e adicionar número se necessário
        $email = $emailBase;
        $contador = 1;
        while (User::where('email', $email)->exists()) {
            $email = $nomeNormalizado . $contador . '@' . $nomeColegioNormalizado . '.edu.mz';
            $contador++;
        }
        
        return $email;
    }

    /**
     * Criar novo funcionário
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'cargo' => 'required|string|max:255',
            'salario_base' => 'nullable|numeric|min:0',
            'numero_dependentes' => 'nullable|integer|min:0',
            'telefone' => 'nullable|string|max:20',
            'endereco' => 'nullable|string',
            'gerir_alunos' => 'boolean',
            'gerir_professores' => 'boolean',
            'gerir_turmas' => 'boolean',
            'gerir_disciplinas' => 'boolean',
            'gerir_mensalidades' => 'boolean',
            'gerir_horarios' => 'boolean',
            'gerir_eventos' => 'boolean',
            'gerir_planos' => 'boolean',
            'gerir_avaliacoes' => 'boolean',
            'gerir_inscricoes' => 'boolean',
            'gerir_recuperacao_senha' => 'boolean',
            'gerir_pagamentos_professores' => 'boolean',
        ]);

        DB::beginTransaction();
        try {
            // Gerar número de funcionário
            $numeroFuncionario = $this->gerarNumeroFuncionario();
            
            // Gerar email
            $email = $this->gerarEmail($request->name);
            
            // Senha padrão
            $senhaPadrao = date('Y');

            // Criar usuário
            $user = $this->userRepository->create([
                'name' => $request->name,
                'email' => $email,
                'password' => $senhaPadrao,
                'tipo' => 'funcionario',
                'must_change_password' => true,
                'is_active' => true,
            ]);

            // Criar funcionário
            $funcionario = Funcionario::create([
                'user_id' => $user->id,
                'numero_funcionario' => $numeroFuncionario,
                'telefone' => $request->telefone,
                'endereco' => $request->endereco,
                'cargo' => trim($request->cargo),
                'salario_base' => $request->salario_base ?? 0,
                'numero_dependentes' => $request->numero_dependentes ?? 0,
                'gerir_alunos' => $request->has('gerir_alunos'),
                'gerir_professores' => $request->has('gerir_professores'),
                'gerir_turmas' => $request->has('gerir_turmas'),
                'gerir_disciplinas' => $request->has('gerir_disciplinas'),
                'gerir_mensalidades' => $request->has('gerir_mensalidades'),
                'gerir_horarios' => $request->has('gerir_horarios'),
                'gerir_eventos' => $request->has('gerir_eventos'),
                'gerir_planos' => $request->has('gerir_planos'),
                'gerir_avaliacoes' => $request->has('gerir_avaliacoes'),
                'gerir_inscricoes' => $request->has('gerir_inscricoes'),
                'gerir_recuperacao_senha' => $request->has('gerir_recuperacao_senha'),
                'gerir_pagamentos_professores' => $request->has('gerir_pagamentos_professores'),
            ]);

            DB::commit();

            return redirect()->route('admin.funcionarios.index')
                ->with('success', "Funcionário criado com sucesso! Número: {$numeroFuncionario}, Senha padrão: {$senhaPadrao}");
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->withInput()
                ->with('error', 'Erro ao criar funcionário: ' . $e->getMessage());
        }
    }

    /**
     * Mostrar detalhes do funcionário
     */
    public function show(Funcionario $funcionario)
    {
        $funcionario->load('user');
        return view('admin.funcionarios.show', compact('funcionario'));
    }

    /**
     * Mostrar formulário de edição
     */
    public function edit(Funcionario $funcionario)
    {
        $funcionario->load('user');
        return view('admin.funcionarios.edit', compact('funcionario'));
    }

    /**
     * Atualizar funcionário
     */
    public function update(Request $request, Funcionario $funcionario)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email,' . $funcionario->user_id,
            'cargo' => 'required|string|max:255',
            'salario_base' => 'nullable|numeric|min:0',
            'numero_dependentes' => 'nullable|integer|min:0',
            'telefone' => 'nullable|string|max:20',
            'endereco' => 'nullable|string',
            'gerir_alunos' => 'boolean',
            'gerir_professores' => 'boolean',
            'gerir_turmas' => 'boolean',
            'gerir_disciplinas' => 'boolean',
            'gerir_mensalidades' => 'boolean',
            'gerir_horarios' => 'boolean',
            'gerir_eventos' => 'boolean',
            'gerir_planos' => 'boolean',
            'gerir_avaliacoes' => 'boolean',
            'gerir_inscricoes' => 'boolean',
            'gerir_recuperacao_senha' => 'boolean',
            'gerir_pagamentos_professores' => 'boolean',
        ]);

        DB::beginTransaction();
        try {
            // Atualizar usuário
            $this->userRepository->update($funcionario->user, [
                'name' => $request->name,
                'email' => $request->email,
            ]);

            // Atualizar funcionário
            $funcionario->update([
                'telefone' => $request->telefone,
                'endereco' => $request->endereco,
                'cargo' => trim($request->cargo),
                'salario_base' => $request->salario_base ?? $funcionario->salario_base ?? 0,
                'numero_dependentes' => $request->numero_dependentes ?? $funcionario->numero_dependentes ?? 0,
                'gerir_alunos' => $request->has('gerir_alunos'),
                'gerir_professores' => $request->has('gerir_professores'),
                'gerir_turmas' => $request->has('gerir_turmas'),
                'gerir_disciplinas' => $request->has('gerir_disciplinas'),
                'gerir_mensalidades' => $request->has('gerir_mensalidades'),
                'gerir_horarios' => $request->has('gerir_horarios'),
                'gerir_eventos' => $request->has('gerir_eventos'),
                'gerir_planos' => $request->has('gerir_planos'),
                'gerir_avaliacoes' => $request->has('gerir_avaliacoes'),
                'gerir_inscricoes' => $request->has('gerir_inscricoes'),
                'gerir_recuperacao_senha' => $request->has('gerir_recuperacao_senha'),
                'gerir_pagamentos_professores' => $request->has('gerir_pagamentos_professores'),
            ]);

            DB::commit();

            return redirect()->route('admin.funcionarios.index')
                ->with('success', 'Funcionário atualizado com sucesso!');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->withInput()
                ->with('error', 'Erro ao atualizar funcionário: ' . $e->getMessage());
        }
    }

    /**
     * Deletar funcionário
     */
    public function destroy(Funcionario $funcionario)
    {
        DB::beginTransaction();
        try {
            $user = $funcionario->user;

            // Hard delete do funcionário (remove o registo da tabela funcionarios)
            $funcionario->forceDelete();

            // Hard delete do user (remove o registo da tabela users)
            if ($user) {
                $user->forceDelete();
            }
            
            DB::commit();

            if (request()->ajax() || request()->wantsJson()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Funcionário eliminado com sucesso!'
                ]);
            }

            return redirect()->route('admin.funcionarios.index')
                ->with('success', 'Funcionário eliminado com sucesso!');
        } catch (\Exception $e) {
            DB::rollBack();

            if (request()->ajax() || request()->wantsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Erro ao eliminar funcionário: ' . $e->getMessage()
                ], 500);
            }

            return redirect()->back()
                ->with('error', 'Erro ao eliminar funcionário: ' . $e->getMessage());
        }
    }

    /**
     * Retorna lista de cargos padrão válidos
     */
    private function cargosPadrao(): array
    {
        return [
            'ADMINISTRAÇÃO',
            'LIMPEZA',
            'SEGURANÇA',
            'PORTARIA',
            'MANUTENÇÃO',
            'COZINHA',
            'MOTORISTA',
            'JARDINAGEM',
            'OUTRO',
        ];
    }

    /**
     * Toggle status do funcionário
     */
    public function toggleStatus(Funcionario $funcionario)
    {
        // Garantir que o relacionamento user está carregado
        if (!$funcionario->relationLoaded('user')) {
            $funcionario->load('user');
        }

        $novoStatus = !$funcionario->user->is_active;
        
        $funcionario->user->update([
            'is_active' => $novoStatus
        ]);

        // Recarregar o modelo para refletir as mudanças
        $funcionario->refresh();
        $funcionario->user->refresh();

        $status = $novoStatus ? 'ativado' : 'desativado';
        return redirect()->back()
            ->with('success', "Funcionário {$status} com sucesso!");
    }
}
