<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\Aluno;
use App\Models\Professor;
use App\Models\Funcionario;
use App\Models\Auditoria;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\DB;

class PasswordResetController extends Controller
{
    public function __construct()
    {
        $this->middleware(['auth', 'role:admin,superadmin,funcionario']);
    }

    /**
     * Verifica se o usuário tem permissão para recuperar senhas
     */
    private function verificarPermissao()
    {
        $user = auth()->user();
        
        // Admin e Superadmin têm todas as permissões
        if ($user->isAdmin() || $user->isSuperadmin()) {
            return true;
        }
        
        // Funcionários precisam ter a permissão específica
        if ($user->isFuncionario()) {
            if (!$user->podeGerir('gerir_recuperacao_senha')) {
                abort(403, 'Acesso negado. Você não tem permissão para recuperar senhas.');
            }
        }
        
        return true;
    }

    /**
     * Mostra a página de busca de usuários para recuperação de senha
     */
    public function index(Request $request)
    {
        $this->verificarPermissao();

        $users = collect();
        $search = $request->get('search');

        if ($search) {
            // Busca por email, nome, código de estudante ou número de funcionário
            $users = User::where(function($query) use ($search) {
                $query->where('email', 'like', "%{$search}%")
                      ->orWhere('name', 'like', "%{$search}%");
            })
            ->whereIn('tipo', ['aluno', 'professor', 'funcionario', 'admin'])
            ->with(['aluno', 'professor', 'funcionario'])
            ->get();

            // Busca também por código de estudante ou número de funcionário
            $alunos = Aluno::where('codigo_estudante', 'like', "%{$search}%")
                          ->with('user')
                          ->get();
            
            $professores = Professor::where('numero_funcionario', 'like', "%{$search}%")
                                  ->with('user')
                                  ->get();
            
            $funcionarios = Funcionario::where('numero_funcionario', 'like', "%{$search}%")
                                     ->with('user')
                                     ->get();

            // Adiciona os usuários encontrados
            foreach ($alunos as $aluno) {
                if ($aluno->user && !$users->contains('id', $aluno->user->id)) {
                    $users->push($aluno->user);
                }
            }

            foreach ($professores as $professor) {
                if ($professor->user && !$users->contains('id', $professor->user->id)) {
                    $users->push($professor->user);
                }
            }

            foreach ($funcionarios as $funcionario) {
                if ($funcionario->user && !$users->contains('id', $funcionario->user->id)) {
                    $users->push($funcionario->user);
                }
            }
        }

        return view('admin.password-reset.index', compact('users', 'search'));
    }

    /**
     * Mostra o formulário para redefinir senha de um usuário específico
     */
    public function show(User $user)
    {
        $this->verificarPermissao();

        // Verifica se o funcionário pode recuperar senha deste tipo de usuário
        $currentUser = auth()->user();
        if ($currentUser->isFuncionario()) {
            // Funcionários podem recuperar senha de alunos, professores e outros funcionários
            if (!in_array($user->tipo, ['aluno', 'professor', 'funcionario'])) {
                abort(403, 'Você só pode recuperar senhas de alunos, professores e funcionários.');
            }
        }

        return view('admin.password-reset.reset', compact('user'));
    }

    /**
     * Processa a redefinição de senha
     */
    public function reset(Request $request, User $user)
    {
        $this->verificarPermissao();

        // Verifica se o funcionário pode recuperar senha deste tipo de usuário
        $currentUser = auth()->user();
        if ($currentUser->isFuncionario()) {
            // Funcionários podem recuperar senha de alunos, professores e outros funcionários
            if (!in_array($user->tipo, ['aluno', 'professor', 'funcionario'])) {
                abort(403, 'Você só pode recuperar senhas de alunos, professores e funcionários.');
            }
        }

        $request->validate([
            'password' => 'required|string|min:8|confirmed',
        ]);

        DB::beginTransaction();
        try {
            $user->password = Hash::make($request->password);
            $user->must_change_password = true; // Força alteração na próxima login
            $user->save();

            // Log de auditoria
            Auditoria::log('PASSWORD_RESET', 'User', $user->id, null, [
                'reset_by' => auth()->id(),
                'reset_by_type' => auth()->user()->tipo,
            ]);

            DB::commit();

            return redirect()->route('admin.password-reset.index')
                ->with('success', "Senha redefinida com sucesso para {$user->name}. O usuário precisará alterar a senha no próximo login.");
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->withInput()
                ->with('error', 'Erro ao redefinir senha: ' . $e->getMessage());
        }
    }
}
