<?php

namespace App\Http\Controllers;

use App\Models\Payment;
use App\Models\Invoice;
use App\Models\Financeiro;
use App\Models\Aluno;
use App\Models\AnoLetivo;
use App\Models\Notification;
use App\Models\User;
use App\Services\FinanceiroService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class PaymentController extends Controller
{
    protected $financeiroService;

    public function __construct(FinanceiroService $financeiroService)
    {
        $this->middleware(['auth', 'role:aluno']);
        $this->financeiroService = $financeiroService;
    }

    /**
     * Mostra a página de pagamento
     */
    public function create(Request $request)
    {
        $user = auth()->user();
        $aluno = $user->aluno;
        $anoLetivoAtivo = AnoLetivo::getAnoLetivoAtual();

        if (!$aluno) {
            return redirect()->route('aluno.dashboard')
                ->withErrors(['error' => 'Perfil de estudante não encontrado.']);
        }

        // Buscar todas as mensalidades (pendentes, vencidas e futuras) para permitir pagamento adiantado
        $dividas = collect();
        $multasPorFinanceiro = [];
        if ($anoLetivoAtivo) {
            // Se há um financeiro_id específico na requisição, buscar apenas esse
            if ($request->has('financeiro_id')) {
                $dividas = $aluno->financeiro()
                    ->where('ano_letivo_id', $anoLetivoAtivo->id)
                    ->where('tipo', 'Mensalidade')
                    ->where('id', $request->financeiro_id)
                    ->where('status', '!=', 'PAGO')
                    ->where('status', '!=', 'CANCELADO')
                    ->orderBy('data_vencimento')
                    ->get();
            } else {
                // Buscar todas as mensalidades não pagas (incluindo futuras) para permitir pagamento adiantado
                $dividas = $aluno->financeiro()
                    ->where('ano_letivo_id', $anoLetivoAtivo->id)
                    ->where('tipo', 'Mensalidade')
                    ->where('status', '!=', 'PAGO')
                    ->where('status', '!=', 'CANCELADO')
                    ->orderBy('data_vencimento')
                    ->get();
            }
            
            // Data limite: 4 dias atrás (após o período de graça - dia 5 é última data sem multa)
            // Última data sem multa: dia 5. A partir do dia 6 começa a multa.
            $dataLimite = now()->subDays(4);
            
            // Calcular multas para cada mensalidade
            foreach ($dividas as $divida) {
                if ($divida->tipo === 'Mensalidade') {
                    $multa = $this->financeiroService->calcularMulta($divida);
                    $valorComMulta = $this->financeiroService->getValorComMulta($divida);
                    $diasAtraso = 0;
                    // Calcular dias de atraso apenas se estiver vencida há mais de 4 dias (a partir do dia 6)
                    if ($divida->data_vencimento <= $dataLimite) {
                        $hoje = now()->startOfDay();
                        $vencimento = \Carbon\Carbon::parse($divida->data_vencimento)->startOfDay();
                        $diasAtraso = $hoje->diffInDays($vencimento);
                    }
                    
                    $multasPorFinanceiro[$divida->id] = [
                        'multa' => $multa,
                        'valor_com_multa' => $valorComMulta,
                        'dias_atraso' => $diasAtraso,
                    ];
                }
            }
        }

        $totalDivida = 0;
        $totalMultas = 0;
        foreach ($dividas as $divida) {
            if ($divida->tipo === 'Mensalidade' && isset($multasPorFinanceiro[$divida->id])) {
                $totalDivida += $multasPorFinanceiro[$divida->id]['valor_com_multa'];
                $totalMultas += $multasPorFinanceiro[$divida->id]['multa'];
            } else {
                $totalDivida += $divida->valor;
            }
        }

        // Buscar turma do aluno no ano letivo ativo
        $turma = null;
        if ($anoLetivoAtivo) {
            $turma = $aluno->turmas()
                ->wherePivot('ano_letivo_id', $anoLetivoAtivo->id)
                ->first();
        }

        return view('aluno.payment.create', compact('aluno', 'dividas', 'multasPorFinanceiro', 'totalDivida', 'totalMultas', 'anoLetivoAtivo', 'turma'));
    }

    /**
     * Processa o pagamento
     */
    public function store(Request $request)
    {
        $request->validate([
            'financeiro_id' => 'required|exists:financeiro,id',
            'metodo' => 'required|in:EMOLA,MPESA',
            'valor' => 'required|numeric|min:0.01',
        ]);

        $user = auth()->user();
        $aluno = $user->aluno;
        $anoLetivoAtivo = AnoLetivo::getAnoLetivoAtual();

        if (!$aluno) {
            return redirect()->back()->withErrors(['error' => 'Perfil de estudante não encontrado.']);
        }

        $financeiro = Financeiro::findOrFail($request->financeiro_id);

        // Verificar se o financeiro pertence ao aluno
        if ($financeiro->aluno_id !== $aluno->id) {
            return redirect()->back()->withErrors(['error' => 'Acesso negado.']);
        }

        // Verificar se já está pago
        if ($financeiro->status === 'PAGO') {
            return redirect()->back()->withErrors(['error' => 'Esta mensalidade já foi paga.']);
        }

        // Validar pagamento sequencial: só pode pagar meses futuros se os anteriores estiverem pagos
        if ($financeiro->tipo === 'Mensalidade') {
            try {
                $this->validarPagamentoSequencial($financeiro);
            } catch (\Exception $e) {
                return redirect()->back()->withErrors(['error' => $e->getMessage()]);
            }
        }

        DB::beginTransaction();
        try {
            // Criar pagamento
            $payment = Payment::create([
                'aluno_id' => $aluno->id,
                'financeiro_id' => $financeiro->id,
                'reference' => Payment::gerarReferencia(),
                'metodo' => $request->metodo,
                'valor' => $request->valor,
                'status' => 'PENDENTE',
            ]);

            // Criar factura
            $invoice = Invoice::create([
                'aluno_id' => $aluno->id,
                'payment_id' => $payment->id,
                'ano_letivo_id' => $anoLetivoAtivo->id,
                'numero_factura' => Invoice::gerarNumeroFactura(),
                'codigo_verificacao' => Invoice::gerarCodigoVerificacao(),
                'valor' => $request->valor,
                'status' => 'PENDENTE',
                'data_emissao' => now(),
                'descricao' => $financeiro->descricao,
                'itens' => [
                    [
                        'descricao' => $financeiro->descricao,
                        'valor' => $request->valor,
                    ]
                ],
            ]);

            $payment->invoice_id = $invoice->id;
            $payment->save();

            // Iniciar processo de pagamento com gateway
            $paymentUrl = $this->iniciarPagamentoGateway($payment, $request->metodo);

            DB::commit();

            return redirect($paymentUrl);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Erro ao processar pagamento: ' . $e->getMessage());
            return redirect()->back()->withErrors(['error' => 'Erro ao processar pagamento. Tente novamente.']);
        }
    }

    /**
     * Callback do gateway de pagamento
     */
    public function callback(Request $request)
    {
        $reference = $request->input('reference');
        $status = $request->input('status');
        $transactionId = $request->input('transaction_id');

        $payment = Payment::where('reference', $reference)->first();

        if (!$payment) {
            return response()->json(['error' => 'Pagamento não encontrado'], 404);
        }

        DB::beginTransaction();
        try {
            $payment->transaction_id = $transactionId;
            $payment->callback_data = $request->all();

            if ($status === 'success' || $status === 'approved') {
                $payment->status = 'APROVADO';
                $payment->data_pagamento = now();

                // Atualizar financeiro
                if ($payment->financeiro) {
                    $this->financeiroService->registrarPagamento(
                        $payment->financeiro,
                        $payment->valor,
                        now()
                    );
                }

                // Atualizar factura
                if ($payment->invoice) {
                    $payment->invoice->status = 'PAGA';
                    $payment->invoice->save();
                }

                // Gerar PDF da factura
                $this->gerarPdfFactura($payment->invoice);
                
                // Criar notificações
                $this->criarNotificacoesPagamento($payment);
            } else {
                $payment->status = 'REJEITADO';
            }

            $payment->save();

            DB::commit();

            return response()->json(['success' => true, 'payment' => $payment]);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Erro no callback de pagamento: ' . $e->getMessage());
            return response()->json(['error' => 'Erro ao processar callback'], 500);
        }
    }

    /**
     * Inicia o pagamento no gateway (Emola ou M-Pesa)
     */
    private function iniciarPagamentoGateway(Payment $payment, string $metodo): string
    {
        // TODO: Implementar integração real com Emola e M-Pesa
        // Por enquanto, retorna uma URL de simulação
        
        if ($metodo === 'MPESA') {
            // Integração M-Pesa
            // Exemplo: usar API do M-Pesa para gerar link de pagamento
            return route('aluno.payment.process', ['payment' => $payment->id, 'metodo' => 'mpesa']);
        } else {
            // Integração Emola
            // Exemplo: usar API do Emola para gerar link de pagamento
            return route('aluno.payment.process', ['payment' => $payment->id, 'metodo' => 'emola']);
        }
    }

    /**
     * Processa o pagamento (simulação)
     */
    public function process(Request $request, Payment $payment)
    {
        $user = auth()->user();
        
        if ($payment->aluno_id !== $user->aluno->id) {
            abort(403);
        }

        return view('aluno.payment.process', compact('payment'));
    }

    /**
     * Confirma pagamento (simulação - para testes)
     */
    public function confirm(Request $request, Payment $payment)
    {
        $user = auth()->user();
        
        if ($payment->aluno_id !== $user->aluno->id) {
            abort(403);
        }

        DB::beginTransaction();
        try {
            $payment->status = 'APROVADO';
            $payment->data_pagamento = now();
            $payment->transaction_id = 'SIM-' . time();
            $payment->save();

            // Atualizar financeiro
            if ($payment->financeiro) {
                $this->financeiroService->registrarPagamento(
                    $payment->financeiro,
                    $payment->valor,
                    now()
                );
            }

            // Atualizar factura
            if ($payment->invoice) {
                $payment->invoice->status = 'PAGA';
                $payment->invoice->save();
            }
            
            // Criar notificações
            $this->criarNotificacoesPagamento($payment);

            DB::commit();

            return redirect()->route('aluno.payment.success', $payment)
                ->with('success', 'Pagamento confirmado com sucesso!');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->withErrors(['error' => 'Erro ao confirmar pagamento.']);
        }
    }

    /**
     * Página de sucesso
     */
    public function success(Payment $payment)
    {
        $user = auth()->user();
        
        if ($payment->aluno_id !== $user->aluno->id) {
            abort(403);
        }

        return view('aluno.payment.success', compact('payment'));
    }

    /**
     * Gera PDF da factura
     */
    private function gerarPdfFactura(Invoice $invoice)
    {
        // TODO: Implementar geração de PDF usando dompdf
        // Por enquanto, apenas marca como gerada
        $invoice->caminho_pdf = 'invoices/' . $invoice->numero_factura . '.pdf';
        $invoice->save();
    }

    /**
     * Criar notificações de pagamento para admin e funcionários
     */
    private function criarNotificacoesPagamento(Payment $payment)
    {
        $payment->load(['aluno.user', 'financeiro']);
        $alunoNome = $payment->aluno->user->name ?? 'Aluno';
        $valor = $payment->valor;
        $metodo = $payment->metodo;
        
        // Link para visualizar o pagamento
        $link = route('financeiro.index', ['ano_letivo_id' => $payment->financeiro->ano_letivo_id ?? null]);
        
        // Notificar todos os admins e superadmins (todos os tipos de pagamento)
        $admins = User::whereIn('tipo', ['admin', 'superadmin'])->get();
        foreach ($admins as $admin) {
            Notification::criarNotificacaoPagamento(
                $admin->id,
                'payment',
                $valor,
                $alunoNome,
                $metodo,
                $link
            );
        }
        
        // Notificar funcionários (apenas M-Pesa e Emola)
        if (in_array($metodo, ['MPESA', 'EMOLA'])) {
            $funcionarios = User::where('tipo', 'funcionario')
                ->whereHas('funcionario', function($q) {
                    $q->where('gerir_mensalidades', true);
                })
                ->get();
            
            foreach ($funcionarios as $funcionario) {
                Notification::criarNotificacaoPagamento(
                    $funcionario->id,
                    'payment',
                    $valor,
                    $alunoNome,
                    $metodo,
                    $link
                );
            }
        }
    }

    /**
     * Valida pagamento sequencial: só pode pagar meses futuros se os anteriores estiverem pagos
     * NÃO permite pagar mês seguinte sem pagar os anteriores, independentemente de estarem vencidos
     */
    private function validarPagamentoSequencial(Financeiro $financeiro)
    {
        // Extrair mês da descrição ou usar mês de vencimento
        $mes = null;
        $ano = null;
        
        // Tentar extrair da descrição
        if (preg_match('/(\d+)\/(\d+)/', $financeiro->descricao, $matches)) {
            $mes = (int) $matches[1];
            $ano = (int) $matches[2];
        } else {
            // Se não encontrar na descrição, usar mês de vencimento
            $mes = (int) $financeiro->data_vencimento->format('m');
            $ano = (int) $financeiro->data_vencimento->format('Y');
        }
        
        if (!$mes || !$ano) {
            return; // Se não conseguir determinar mês/ano, não validar
        }
        
        // Verificar se há meses anteriores não pagos
        // NÃO permitir pagar mês seguinte sem pagar os anteriores, independentemente de estarem vencidos
        for ($m = 1; $m < $mes; $m++) {
            $mensalidadeAnterior = Financeiro::where('aluno_id', $financeiro->aluno_id)
                ->where('ano_letivo_id', $financeiro->ano_letivo_id)
                ->where('tipo', 'Mensalidade')
                ->where(function($query) use ($m, $ano) {
                    $query->where('descricao', 'like', "%{$m}/{$ano}%")
                          ->orWhere(function($q) use ($m, $ano) {
                              // Também verificar por mês de vencimento
                              $q->whereMonth('data_vencimento', $m)
                                ->whereYear('data_vencimento', $ano);
                          });
                })
                ->first();
            
            // Se a mensalidade anterior existe e não está paga, bloquear pagamento
            if ($mensalidadeAnterior && $mensalidadeAnterior->status !== 'PAGO') {
                $meses = ['', 'Janeiro', 'Fevereiro', 'Março', 'Abril', 'Maio', 'Junho', 
                         'Julho', 'Agosto', 'Setembro', 'Outubro', 'Novembro', 'Dezembro'];
                $mesNomeAnterior = $meses[$m] ?? $m;
                $mesNomeAtual = $meses[$mes] ?? $mes;
                throw new \Exception("Não é possível pagar a mensalidade de {$mesNomeAtual}/{$ano} sem antes pagar a mensalidade de {$mesNomeAnterior}/{$ano}.");
            }
        }
    }
}
