<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\AnoLetivo;
use App\Models\Horario;
use App\Models\Turma;
use App\Models\Disciplina;
use App\Models\Avaliacao;
use Illuminate\Support\Facades\DB;

class ProfessorController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Dashboard do professor
     */
    public function dashboard()
    {
        $user = auth()->user();
        $professor = $user->professor;
        $anoLetivoAtivo = AnoLetivo::where('ativo', true)->first();

        if (!$professor) {
            return redirect()->route('dashboard')
                ->withErrors(['error' => 'Seu perfil de professor não está completo.']);
        }

        // Buscar turmas e disciplinas do professor
        $alocacoes = [];
        if ($anoLetivoAtivo) {
            $alocacoes = DB::table('turma_disciplina_professor')
                ->where('turma_disciplina_professor.professor_id', $professor->id)
                ->where('turma_disciplina_professor.ano_letivo_id', $anoLetivoAtivo->id)
                ->join('turmas', 'turma_disciplina_professor.turma_id', '=', 'turmas.id')
                ->join('disciplinas', 'turma_disciplina_professor.disciplina_id', '=', 'disciplinas.id')
                ->select(
                    'turmas.id as turma_id',
                    'turmas.nome as turma_nome',
                    'turmas.codigo as turma_codigo',
                    'disciplinas.id as disciplina_id',
                    'disciplinas.nome as disciplina_nome'
                )
                ->get();
        }

        // Contar avaliações lançadas
        $totalAvaliacoes = 0;
        if ($anoLetivoAtivo) {
            $totalAvaliacoes = Avaliacao::where('ano_letivo_id', $anoLetivoAtivo->id)
                ->whereIn('turma_id', $alocacoes->pluck('turma_id'))
                ->whereIn('disciplina_id', $alocacoes->pluck('disciplina_id'))
                ->count();
        }

        // Buscar horários do professor
        $horarios = collect();
        if ($anoLetivoAtivo) {
            $horarios = Horario::where('ano_letivo_id', $anoLetivoAtivo->id)
                ->where('professor_id', $professor->id)
                ->with(['turma', 'disciplina'])
                ->orderBy('dia_semana')
                ->orderBy('hora_inicio')
                ->get();
        }

        return view('professor.dashboard', compact(
            'professor',
            'anoLetivoAtivo',
            'alocacoes',
            'totalAvaliacoes',
            'horarios'
        ));
    }

    /**
     * Visualizar horários do professor
     */
    public function horarios()
    {
        $user = auth()->user();
        
        if (!$user->isProfessor()) {
            abort(403, 'Acesso negado. Apenas professores podem acessar esta área.');
        }
        
        $professor = $user->professor;
        $anoLetivoAtivo = AnoLetivo::where('ativo', true)->first();

        if (!$professor) {
            return redirect()->route('dashboard')
                ->withErrors(['error' => 'Seu perfil de professor não está completo.']);
        }

        $horarios = collect();
        if ($anoLetivoAtivo) {
            $horarios = Horario::where('ano_letivo_id', $anoLetivoAtivo->id)
                ->where('professor_id', $professor->id)
                ->with(['turma', 'disciplina'])
                ->orderBy('dia_semana')
                ->orderBy('hora_inicio')
                ->get();
        }

        // Organizar por dia da semana
        $diasSemana = ['Segunda', 'Terça', 'Quarta', 'Quinta', 'Sexta'];
        $horariosPorDia = [];
        
        foreach ($diasSemana as $dia) {
            $horariosPorDia[$dia] = $horarios->filter(function($h) use ($dia) {
                return $h->dia_semana === $dia;
            })->sortBy('hora_inicio')->values();
        }

        return view('professor.horarios', compact('professor', 'anoLetivoAtivo', 'horariosPorDia', 'diasSemana'));
    }

    /**
     * Visualizar turmas e disciplinas do professor
     */
    public function turmas()
    {
        $user = auth()->user();
        
        if (!$user->isProfessor()) {
            abort(403, 'Acesso negado. Apenas professores podem acessar esta área.');
        }
        
        $professor = $user->professor;
        $anoLetivoAtivo = AnoLetivo::where('ativo', true)->first();

        if (!$professor) {
            return redirect()->route('dashboard')
                ->withErrors(['error' => 'Seu perfil de professor não está completo.']);
        }

        $alocacoes = [];
        if ($anoLetivoAtivo) {
            $alocacoes = DB::table('turma_disciplina_professor')
                ->where('turma_disciplina_professor.professor_id', $professor->id)
                ->where('turma_disciplina_professor.ano_letivo_id', $anoLetivoAtivo->id)
                ->join('turmas', 'turma_disciplina_professor.turma_id', '=', 'turmas.id')
                ->join('disciplinas', 'turma_disciplina_professor.disciplina_id', '=', 'disciplinas.id')
                ->whereNull('turmas.deleted_at')
                ->whereNull('disciplinas.deleted_at')
                ->select(
                    'turmas.id as turma_id',
                    'turmas.nome as turma_nome',
                    'turmas.codigo as turma_codigo',
                    'turmas.nivel_ensino',
                    'disciplinas.id as disciplina_id',
                    'disciplinas.nome as disciplina_nome',
                    'disciplinas.modelo_avaliacao'
                )
                ->orderBy('turmas.codigo')
                ->orderBy('turmas.nome')
                ->orderBy('disciplinas.nome')
                ->get();
        }

        // Para cada alocação, buscar número de alunos
        foreach ($alocacoes as $alocacao) {
            $turma = Turma::find($alocacao->turma_id);
            if ($turma && $anoLetivoAtivo) {
                $alocacao->total_alunos = $turma->alunos()
                    ->wherePivot('ano_letivo_id', $anoLetivoAtivo->id)
                    ->count();
            } else {
                $alocacao->total_alunos = 0;
            }
        }
        
        // Agrupar alocações por classe (código) para facilitar visualização
        $alocacoesAgrupadas = $alocacoes->groupBy('turma_codigo');

        return view('professor.turmas', compact('professor', 'anoLetivoAtivo', 'alocacoes', 'alocacoesAgrupadas'));
    }

    /**
     * Visualizar perfil do professor
     */
    public function perfil()
    {
        $user = auth()->user();
        
        if (!$user->isProfessor()) {
            abort(403, 'Acesso negado. Apenas professores podem acessar esta área.');
        }
        
        $professor = $user->professor;

        if (!$professor) {
            return redirect()->route('dashboard')
                ->withErrors(['error' => 'Seu perfil de professor não está completo.']);
        }

        $professor->load('user');
        
        // Buscar disciplinas que o professor leciona
        $anoLetivo = AnoLetivo::where('ativo', true)->first();
        $disciplinas = DB::table('turma_disciplina_professor')
            ->where('turma_disciplina_professor.professor_id', $professor->id)
            ->when($anoLetivo, function($query) use ($anoLetivo) {
                $query->where('turma_disciplina_professor.ano_letivo_id', $anoLetivo->id);
            })
            ->join('disciplinas', 'turma_disciplina_professor.disciplina_id', '=', 'disciplinas.id')
            ->select('disciplinas.id', 'disciplinas.nome', 'disciplinas.modelo_avaliacao')
            ->distinct()
            ->orderBy('disciplinas.nome')
            ->get();

        return view('professor.perfil', compact('professor', 'disciplinas', 'anoLetivo'));
    }
}

