<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Aluno;
use App\Models\User;
use App\Repositories\AlunoRepository;
use App\Services\CodigoEstudanteService;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class PublicInscricaoController extends Controller
{
    protected $alunoRepository;

    public function __construct(AlunoRepository $alunoRepository)
    {
        $this->alunoRepository = $alunoRepository;
    }

    /**
     * Mostrar formulário público de inscrição
     */
    public function showForm()
    {
        return view('public.inscricao');
    }

    /**
     * Processar inscrição pública
     */
    public function store(Request $request)
    {
        // Validação dos dados
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'email' => 'nullable|email|unique:users,email',
            'telefone' => 'nullable|string',
            'endereco' => 'nullable|string',
            'data_nascimento' => 'nullable|date',
            'local_nascimento' => 'nullable|string',
            'naturalidade' => 'nullable|string',
            'distrito_natural' => 'nullable|string',
            'provincia_natural' => 'nullable|string',
            'num_irmaos_mais_velhos' => 'nullable|integer|min:0',
            'num_irmaos_mais_novos' => 'nullable|integer|min:0',
            'num_irmaos_mesma_idade' => 'nullable|integer|min:0',
            'num_documento' => 'nullable|string',
            'tipo_documento' => 'nullable|in:BI,Cédula,Passaporte',
            'foto' => 'nullable|image|mimes:jpeg,jpg,png|max:2048',
            'foto_base64' => 'nullable|string',
            'local_emissao' => 'nullable|string',
            'data_emissao' => 'nullable|date',
            'nome_pai' => 'nullable|string',
            'local_trabalho_pai' => 'nullable|string',
            'endereco_pai_bairro' => 'nullable|string',
            'endereco_pai_rua' => 'nullable|string',
            'endereco_pai_numero' => 'nullable|string',
            'endereco_pai_quarteirao' => 'nullable|string',
            'endereco_pai_flat' => 'nullable|string',
            'telefone_pai' => 'nullable|string',
            'bi_pai' => 'nullable|string',
            'nome_mae' => 'nullable|string',
            'local_trabalho_mae' => 'nullable|string',
            'endereco_mae_bairro' => 'nullable|string',
            'endereco_mae_rua' => 'nullable|string',
            'endereco_mae_numero' => 'nullable|string',
            'endereco_mae_quarteirao' => 'nullable|string',
            'endereco_mae_flat' => 'nullable|string',
            'telefone_mae' => 'nullable|string',
            'bi_mae' => 'nullable|string',
            'genero' => 'nullable|in:M,F,Outro',
            'plano_saude' => 'nullable|string',
            'problema_saude_cronico' => 'nullable|string',
            'alergias' => 'nullable|string',
            'diagnostico_deficiencia' => 'nullable|string',
            'dificuldade_motora_auditiva_visual_fala_emocional' => 'nullable|string',
            'tratamento_medico' => 'nullable|string',
            'medicacao' => 'nullable|string',
            'vacinas_em_dia' => 'nullable|boolean',
            'contato_emergencia' => 'nullable|string',
            'outras_informacoes_saude' => 'nullable|string',
            'individuo_autorizado_1_nome' => 'nullable|string',
            'individuo_autorizado_1_telefone' => 'nullable|string',
            'individuo_autorizado_2_nome' => 'nullable|string',
            'individuo_autorizado_2_telefone' => 'nullable|string',
            'individuo_autorizado_3_nome' => 'nullable|string',
            'individuo_autorizado_3_telefone' => 'nullable|string',
            'nome_encarregado' => 'required|string',
            'telefone_encarregado' => 'required|string',
            'endereco_encarregado' => 'nullable|string',
            'profissao_encarregado' => 'nullable|string',
            'bi_encarregado' => 'nullable|string',
            'grau_parentesco' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        DB::beginTransaction();
        try {
            $dados = $request->except(['foto', 'foto_base64', '_token']);
            
            // Gerar email automático se não fornecido
            $email = $dados['email'] ?? null;
            if (!$email) {
                $nomeBase = $dados['name'];
                $nomeBase = mb_strtolower($nomeBase, 'UTF-8');
                $nomeBase = preg_replace('/[^a-z0-9\s]/', '', $nomeBase);
                $nomeBase = preg_replace('/\s+/', '', $nomeBase);
                $nomeBase = substr($nomeBase, 0, 20);
                
                if (empty($nomeBase)) {
                    $nomeBase = 'aluno';
                }
                
                $email = $nomeBase . '_' . time() . '@aluno.sge.pt';
                
                $contador = 1;
                while (User::where('email', $email)->exists()) {
                    $email = $nomeBase . '_' . time() . '_' . $contador . '@aluno.sge.pt';
                    $contador++;
                }
            }
            
            // Criar usuário
            $user = User::create([
                'name' => $dados['name'],
                'email' => $email,
                'password' => Hash::make(date('Y')), // Senha padrão: ano atual
                'tipo' => 'aluno',
                'must_change_password' => true,
                'is_active' => false, // Inativo até aprovação
            ]);
            
            // Gerar código de estudante
            $anoIngresso = date('Y');
            $codigoEstudante = CodigoEstudanteService::gerar($anoIngresso);
            
            // Processar foto
            if ($request->hasFile('foto')) {
                $arquivo = $request->file('foto');
                $nomeArquivo = time() . '_' . uniqid() . '.' . $arquivo->getClientOriginalExtension();
                $caminho = $arquivo->storeAs('fotos/alunos', $nomeArquivo, 'public');
                $dados['foto'] = $caminho;
            } elseif ($request->filled('foto_base64')) {
                $imageData = $request->foto_base64;
                if (preg_match('/^data:image\/(\w+);base64,/', $imageData, $type)) {
                    $imageData = substr($imageData, strpos($imageData, ',') + 1);
                    $type = strtolower($type[1]);
                    
                    if (in_array($type, ['jpg', 'jpeg', 'png'])) {
                        $imageData = base64_decode($imageData);
                        $nomeArquivo = time() . '_' . uniqid() . '.' . $type;
                        $caminho = 'fotos/alunos/' . $nomeArquivo;
                        Storage::disk('public')->put($caminho, $imageData);
                        $dados['foto'] = $caminho;
                    }
                }
            }
            
            // Construir data_nascimento
            if (!$request->filled('data_nascimento') && $request->filled('dia_nascimento') && $request->filled('mes_nascimento') && $request->filled('ano_nascimento')) {
                $dia = str_pad($request->dia_nascimento, 2, '0', STR_PAD_LEFT);
                $mes = str_pad($request->mes_nascimento, 2, '0', STR_PAD_LEFT);
                $ano = $request->ano_nascimento;
                $dados['data_nascimento'] = "{$ano}-{$mes}-{$dia}";
            }
            
            // Criar aluno com status PENDENTE
            $aluno = Aluno::create(array_merge($dados, [
                'user_id' => $user->id,
                'codigo_estudante' => $codigoEstudante,
                'status_inscricao' => 'PENDENTE',
                'data_submissao' => now(),
                'pago' => false,
            ]));
            
            DB::commit();
            
            return redirect()->route('public.inscricao.success')
                ->with('success', 'Inscrição submetida com sucesso! Aguarde a aprovação da administração.');
                
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Erro ao processar inscrição: ' . $e->getMessage())->withInput();
        }
    }

    /**
     * Página de sucesso após submissão
     */
    public function success()
    {
        return view('public.inscricao-success');
    }
}
