<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\Configuracao;
use App\Models\Tema;
use App\Models\AnoLetivo;
use App\Models\Trimestre;
use App\Models\Disciplina;
use App\Models\Auditoria;
use App\Repositories\UserRepository;
use App\Services\GoogleDriveService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Log;
use ZipArchive;

class SuperadminController extends Controller
{
    protected $userRepository;

    public function __construct(UserRepository $userRepository)
    {
        $this->middleware(['auth', 'role:superadmin']);
        $this->userRepository = $userRepository;
    }

    // Gestão de Admins
    public function indexAdmins()
    {
        // Buscar admins e também funcionários/professores que foram criados como admin
        $admins = User::whereIn('tipo', ['admin', 'funcionario', 'professor'])
            ->orderBy('tipo')
            ->orderBy('name')
            ->get();
        return view('superadmin.admins.index', compact('admins'));
    }

    public function toggleBlockAdmin(User $user)
    {
        if (!in_array($user->tipo, ['admin', 'funcionario', 'professor'])) {
            abort(403);
        }
        
        $dadosAntigos = ['is_active' => $user->is_active];
        $user->is_active = !$user->is_active;
        $user->save();
        
        Auditoria::log('UPDATE', 'User', $user->id, $dadosAntigos, ['is_active' => $user->is_active]);
        
        return redirect()->route('superadmin.admins.index')->with('success', 
            $user->is_active ? 'Usuário desbloqueado com sucesso!' : 'Usuário bloqueado com sucesso!'
        );
    }

    public function createAdmin()
    {
        return view('superadmin.admins.create');
    }

    public function storeAdmin(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email',
            'password' => 'required|string|min:8',
        ]);

        $user = $this->userRepository->create([
            'name' => $request->name,
            'email' => $request->email,
            'password' => $request->password,
            'tipo' => 'admin',
            'must_change_password' => false,
            'is_active' => true,
        ]);

        // Log de auditoria
        Auditoria::log('CREATE', 'User', $user->id, null, ['name' => $user->name, 'email' => $user->email, 'tipo' => 'admin']);

        return redirect()->route('superadmin.admins.index')->with('success', 'Admin criado com sucesso!');
    }

    public function editAdmin(User $user)
    {
        if (!in_array($user->tipo, ['admin', 'funcionario', 'professor'])) {
            abort(403);
        }
        return view('superadmin.admins.edit', compact('user'));
    }

    public function updateAdmin(Request $request, User $user)
    {
        if (!in_array($user->tipo, ['admin', 'funcionario', 'professor'])) {
            abort(403);
        }

        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email,' . $user->id,
            'password' => 'nullable|string|min:8',
            'is_active' => 'boolean',
            'tipo' => 'required|in:admin,funcionario,professor',
        ]);

        $dados = $request->only(['name', 'email', 'is_active', 'tipo']);
        if ($request->filled('password')) {
            $dados['password'] = $request->password;
        }

        $dadosAntigos = [
            'name' => $user->name,
            'email' => $user->email,
            'is_active' => $user->is_active,
            'tipo' => $user->tipo,
        ];
        
        $this->userRepository->update($user, $dados);
        
        // Log de auditoria
        Auditoria::log('UPDATE', 'User', $user->id, $dadosAntigos, $dados);

        return redirect()->route('superadmin.admins.index')->with('success', 'Admin atualizado com sucesso!');
    }

    public function destroyAdmin(User $user)
    {
        if (!in_array($user->tipo, ['admin', 'funcionario', 'professor'])) {
            abort(403);
        }
        
        // Log de auditoria antes de deletar
        Auditoria::log('DELETE', 'User', $user->id, ['name' => $user->name, 'email' => $user->email, 'tipo' => $user->tipo], null);
        
        $user->delete();
        return redirect()->route('superadmin.admins.index')->with('success', 'Usuário eliminado com sucesso!');
    }

    // Configurações do Sistema
    public function configuracoes()
    {
        $modeloAvaliacao = Configuracao::get('modelo_avaliacao_global', 'NACIONAL');
        $anoLetivoAtivo = AnoLetivo::where('ativo', true)->first();
        $temaAtivo = Tema::getTemaAtivo();
        
        // Buscar configurações por trimestre
        $trimestres = $anoLetivoAtivo ? $anoLetivoAtivo->trimestres()->orderBy('numero')->get() : collect();
        $lancamentoPorTrimestre = Configuracao::get('lancamento_notas_trimestres', []);
        if (is_string($lancamentoPorTrimestre)) {
            $lancamentoPorTrimestre = json_decode($lancamentoPorTrimestre, true) ?? [];
        }
        
        // Para cada trimestre, verificar se está liberado (padrão: true se não especificado)
        foreach ($trimestres as $trimestre) {
            if (!isset($lancamentoPorTrimestre[$trimestre->id])) {
                $lancamentoPorTrimestre[$trimestre->id] = true; // Padrão: liberado
            }
        }

        return view('superadmin.configuracoes', compact('modeloAvaliacao', 'anoLetivoAtivo', 'temaAtivo', 'trimestres', 'lancamentoPorTrimestre'));
    }

    public function configuracoesCabecalho()
    {
        $republica = Configuracao::get('republica', 'REPÚBLICA DE MOÇAMBIQUE');
        $provincia = Configuracao::get('provincia', 'PROVINCIAL DA ZAMBÉZIA');
        $distrito = Configuracao::get('distrito', 'GOVERNO DO DISTRITO DE MOCUBA');
        $escola = Configuracao::get('escola', 'ESCOLA PRIMARIA E COMPLETA SGE');
        $logotipo = Configuracao::get('logotipo_escola', null);

        return view('superadmin.configuracoes-cabecalho', compact('republica', 'provincia', 'distrito', 'escola', 'logotipo'));
    }

    public function updateConfiguracoesCabecalho(Request $request)
    {
        $request->validate([
            'republica' => 'required|string|max:255',
            'provincia' => 'required|string|max:255',
            'distrito' => 'required|string|max:255',
            'escola' => 'required|string|max:255',
            'logotipo' => 'nullable|image|mimes:jpeg,jpg,png,gif|max:2048',
        ]);

        Configuracao::set('republica', $request->republica, 'string', 'República de Moçambique');
        Configuracao::set('provincia', $request->provincia, 'string', 'Província');
        Configuracao::set('distrito', $request->distrito, 'string', 'Distrito');
        Configuracao::set('escola', $request->escola, 'string', 'Nome da Escola');

        // Processar upload do logotipo
        if ($request->hasFile('logotipo')) {
            // Deletar logotipo antigo se existir
            $logotipoAntigo = Configuracao::get('logotipo_escola', null);
            if ($logotipoAntigo && Storage::disk('public')->exists($logotipoAntigo)) {
                Storage::disk('public')->delete($logotipoAntigo);
            }
            
            $arquivo = $request->file('logotipo');
            $nomeArquivo = 'logotipo_' . time() . '_' . uniqid() . '.' . $arquivo->getClientOriginalExtension();
            $caminho = $arquivo->storeAs('configuracoes', $nomeArquivo, 'public');
            Configuracao::set('logotipo_escola', $caminho, 'string', 'Logotipo da Escola');
        }

        return redirect()->route('superadmin.configuracoes.cabecalho')->with('success', 'Configurações do cabeçalho atualizadas com sucesso!');
    }

    public function updateConfiguracoes(Request $request)
    {
        $request->validate([
            'ano_letivo_id' => 'nullable|exists:ano_letivo,id',
            'cor_principal' => 'required|string',
            'cor_secundaria' => 'nullable|string',
            'trimestres' => 'nullable|array',
            'trimestres.*' => 'nullable|boolean',
        ]);

        // Modelo de avaliação fixo no sistema nacional
        Configuracao::set('modelo_avaliacao_global', 'NACIONAL', 'string', 'Modelo de avaliação global do sistema');

        // Atualiza controle de lançamento de notas por trimestre
        if ($request->has('trimestres')) {
            // Buscar todos os trimestres do ano letivo ativo
            $anoLetivoAtivo = AnoLetivo::where('ativo', true)->first();
            if ($anoLetivoAtivo) {
                $trimestres = $anoLetivoAtivo->trimestres()->orderBy('numero')->get();
                
                // Com o campo hidden, todos os trimestres sempre estarão no array
                // Se o valor é "1", está marcado (liberado = true)
                // Se o valor é "0" ou não existe, está desmarcado (liberado = false)
                $lancamentoPorTrimestre = [];
                foreach ($trimestres as $trimestre) {
                    // Verifica se o valor é "1" (checkbox marcado)
                    $lancamentoPorTrimestre[$trimestre->id] = isset($request->trimestres[$trimestre->id]) && $request->trimestres[$trimestre->id] == '1';
                }
                
                Configuracao::set('lancamento_notas_trimestres', $lancamentoPorTrimestre, 'json', 'Controla se o lançamento de notas está liberado (true) ou trancado (false) por trimestre');
            }
        }

        // Atualiza ano letivo ativo
        if ($request->ano_letivo_id) {
            AnoLetivo::where('ativo', true)->update(['ativo' => false]);
            AnoLetivo::where('id', $request->ano_letivo_id)->update(['ativo' => true]);
        }

        // Atualiza tema
        $tema = Tema::where('ativo', true)->first();
        if ($tema) {
            $tema->update([
                'cor_principal' => $request->cor_principal,
                'cor_secundaria' => $request->cor_secundaria,
            ]);
        } else {
            Tema::create([
                'nome' => 'Tema Principal',
                'cor_principal' => $request->cor_principal,
                'cor_secundaria' => $request->cor_secundaria,
                'ativo' => true,
            ]);
        }

        return redirect()->route('superadmin.configuracoes')->with('success', 'Configurações atualizadas com sucesso!');
    }

    // Gestão de Trimestres
    public function trimestres()
    {
        $anoLetivo = AnoLetivo::where('ativo', true)->first();
        $trimestres = $anoLetivo ? $anoLetivo->trimestres : collect();
        return view('superadmin.trimestres.index', compact('trimestres', 'anoLetivo'));
    }

    public function storeTrimestre(Request $request)
    {
        $request->validate([
            'ano_letivo_id' => 'required|exists:ano_letivo,id',
            'nome' => 'required|string',
            'numero' => 'required|integer|min:1|max:3',
            'data_inicio' => 'required|date',
            'data_fim' => 'required|date|after:data_inicio',
        ]);

        Trimestre::create($request->all());

        return redirect()->route('superadmin.trimestres')->with('success', 'Trimestre criado com sucesso!');
    }

    // Logs e Auditoria
    public function logs(Request $request)
    {
        $query = Auditoria::with('user');
        
        // Filtro por ação
        if ($request->filled('acao')) {
            $query->where('acao', $request->acao);
        }
        
        // Filtro por usuário
        if ($request->filled('usuario')) {
            $query->whereHas('user', function($q) use ($request) {
                $q->where('name', 'like', '%' . $request->usuario . '%')
                  ->orWhere('email', 'like', '%' . $request->usuario . '%');
            });
        }
        
        // Filtro por data
        if ($request->filled('data_inicio')) {
            $query->whereDate('created_at', '>=', $request->data_inicio);
        }
        
        if ($request->filled('data_fim')) {
            $query->whereDate('created_at', '<=', $request->data_fim);
        }
        
        $logs = $query->orderBy('created_at', 'desc')->paginate(50);
        
        return view('superadmin.logs.index', compact('logs'));
    }

    public function logsErros()
    {
        $logFile = storage_path('logs/laravel.log');
        $erros = [];
        
        if (File::exists($logFile)) {
            $content = File::get($logFile);
            // Buscar linhas com ERROR
            $lines = explode("\n", $content);
            $currentError = null;
            
            foreach ($lines as $line) {
                if (stripos($line, '[ERROR]') !== false || stripos($line, 'ERROR:') !== false) {
                    if ($currentError) {
                        $erros[] = $currentError;
                    }
                    $currentError = $line;
                } elseif ($currentError && (stripos($line, 'Stack trace') !== false || stripos($line, 'at ') !== false || trim($line) !== '')) {
                    $currentError .= "\n" . $line;
                } elseif (trim($line) === '' && $currentError) {
                    $erros[] = $currentError;
                    $currentError = null;
                }
            }
            
            if ($currentError) {
                $erros[] = $currentError;
            }
            
            // Reverter para mostrar os mais recentes primeiro
            $erros = array_reverse($erros);
            $erros = array_slice($erros, -100); // Últimos 100 erros
        }
        
        return view('superadmin.logs.erros', compact('erros'));
    }

    public function logsDetalhes($id)
    {
        $log = Auditoria::with('user')->findOrFail($id);
        return view('superadmin.logs.detalhes', compact('log'));
    }

    // Sistema de Backup
    public function backups()
    {
        $backupDir = storage_path('app/backups');
        $backups = [];
        $tamanhoTotal = 0;
        
        if (File::isDirectory($backupDir)) {
            $files = File::files($backupDir);
            foreach ($files as $file) {
                $ext = strtolower($file->getExtension());
                $nomeArquivo = $file->getFilename();
                $tipo = 'database';
                
                if ($ext === 'zip') {
                    if (stripos($nomeArquivo, 'COMPLETO') !== false || stripos($nomeArquivo, 'completo') !== false || stripos($nomeArquivo, 'full') !== false) {
                        $tipo = 'completo';
                    } elseif (stripos($nomeArquivo, 'ARQUIVOS') !== false || stripos($nomeArquivo, 'files') !== false) {
                        $tipo = 'files';
                    } else {
                        $tipo = 'files'; // ZIP sem indicação específica
                    }
                } elseif ($ext === 'sql') {
                    $tipo = 'database';
                }
                
                $tamanho = $file->getSize();
                $tamanhoTotal += $tamanho;
                
                $backups[] = [
                    'nome' => $nomeArquivo,
                    'tamanho' => $tamanho,
                    'data' => date('Y-m-d H:i:s', $file->getMTime()),
                    'tipo' => $tipo,
                ];
            }
            
            // Ordenar por data (mais recente primeiro)
            usort($backups, function($a, $b) {
                return strtotime($b['data']) - strtotime($a['data']);
            });
        }
        
        // Estatísticas de armazenamento de arquivos
        $estatisticas = $this->calcularEstatisticasArmazenamento();
        
        return view('superadmin.backups.index', compact('backups', 'tamanhoTotal', 'estatisticas'));
    }

    private function calcularEstatisticasArmazenamento()
    {
        $publicDir = storage_path('app/public');
        $estatisticas = [
            'fotos_alunos' => ['tamanho' => 0, 'quantidade' => 0],
            'tpcs' => ['tamanho' => 0, 'quantidade' => 0],
            'planos' => ['tamanho' => 0, 'quantidade' => 0],
            'configuracoes' => ['tamanho' => 0, 'quantidade' => 0],
            'total' => ['tamanho' => 0, 'quantidade' => 0],
        ];
        
        if (File::isDirectory($publicDir)) {
            // Fotos de alunos
            $fotosDir = $publicDir . DIRECTORY_SEPARATOR . 'fotos' . DIRECTORY_SEPARATOR . 'alunos';
            if (File::isDirectory($fotosDir)) {
                $fotos = File::allFiles($fotosDir);
                foreach ($fotos as $foto) {
                    $estatisticas['fotos_alunos']['tamanho'] += $foto->getSize();
                    $estatisticas['fotos_alunos']['quantidade']++;
                }
            }
            
            // TPCs
            $tpcsDir = $publicDir . DIRECTORY_SEPARATOR . 'tpcs';
            if (File::isDirectory($tpcsDir)) {
                $tpcs = File::allFiles($tpcsDir);
                foreach ($tpcs as $tpc) {
                    $estatisticas['tpcs']['tamanho'] += $tpc->getSize();
                    $estatisticas['tpcs']['quantidade']++;
                }
            }
            
            // Planos trimestrais
            $planosDir = $publicDir . DIRECTORY_SEPARATOR . 'planos-trimestrais';
            if (File::isDirectory($planosDir)) {
                $planos = File::allFiles($planosDir);
                foreach ($planos as $plano) {
                    $estatisticas['planos']['tamanho'] += $plano->getSize();
                    $estatisticas['planos']['quantidade']++;
                }
            }
            
            // Configurações
            $configDir = $publicDir . DIRECTORY_SEPARATOR . 'configuracoes';
            if (File::isDirectory($configDir)) {
                $configs = File::allFiles($configDir);
                foreach ($configs as $config) {
                    $estatisticas['configuracoes']['tamanho'] += $config->getSize();
                    $estatisticas['configuracoes']['quantidade']++;
                }
            }
            
            // Total
            $estatisticas['total']['tamanho'] = 
                $estatisticas['fotos_alunos']['tamanho'] +
                $estatisticas['tpcs']['tamanho'] +
                $estatisticas['planos']['tamanho'] +
                $estatisticas['configuracoes']['tamanho'];
            
            $estatisticas['total']['quantidade'] = 
                $estatisticas['fotos_alunos']['quantidade'] +
                $estatisticas['tpcs']['quantidade'] +
                $estatisticas['planos']['quantidade'] +
                $estatisticas['configuracoes']['quantidade'];
        }
        
        return $estatisticas;
    }

    public function limparBackupsAntigos(Request $request)
    {
        try {
            $dias = $request->input('dias', 30); // Padrão: 30 dias
            $backupDir = storage_path('app/backups');
            $deletados = 0;
            $espacoLiberado = 0;
            
            if (!File::isDirectory($backupDir)) {
                return redirect()->route('superadmin.backups.index')->withErrors(['error' => 'Diretório de backups não encontrado.']);
            }
            
            $files = File::files($backupDir);
            $limiteData = strtotime("-{$dias} days");
            
            foreach ($files as $file) {
                $dataModificacao = $file->getMTime();
                if ($dataModificacao < $limiteData) {
                    $tamanho = $file->getSize();
                    File::delete($file->getRealPath());
                    $deletados++;
                    $espacoLiberado += $tamanho;
                }
            }
            
            Auditoria::log('BACKUP_CLEANUP', 'Backup', null, null, [
                'dias' => $dias,
                'deletados' => $deletados,
                'espaco_liberado_mb' => round($espacoLiberado / 1024 / 1024, 2)
            ]);
            
            $mensagem = "Limpeza concluída! {$deletados} backup(s) antigo(s) deletado(s). ";
            $mensagem .= "Espaço liberado: " . number_format($espacoLiberado / 1024 / 1024, 2) . " MB.";
            
            return redirect()->route('superadmin.backups.index')->with('success', $mensagem);
            
        } catch (\Exception $e) {
            Log::error('Erro ao limpar backups antigos: ' . $e->getMessage());
            return redirect()->route('superadmin.backups.index')->withErrors(['error' => 'Erro ao limpar backups: ' . $e->getMessage()]);
        }
    }

    public function criarBackup(Request $request)
    {
        try {
            // Debug: verificar se a requisição chegou
            Log::info('Requisição de backup recebida', [
                'tipo' => $request->input('tipo'),
                'all' => $request->all(),
                'method' => $request->method()
            ]);
            
            $tipo = $request->input('tipo', 'database'); // database, files, completo
            
            // Validar tipo
            if (!in_array($tipo, ['database', 'files', 'completo'])) {
                Log::error('Tipo de backup inválido', ['tipo' => $tipo]);
                return redirect()->route('superadmin.backups.index')->withErrors(['error' => 'Tipo de backup inválido: ' . $tipo]);
            }
            
            $backupDir = storage_path('app/backups');
            if (!File::isDirectory($backupDir)) {
                File::makeDirectory($backupDir, 0755, true);
                Log::info('Diretório de backup criado', ['dir' => $backupDir]);
            }
            
            // Obter nome da escola
            $nomeEscola = Configuracao::get('escola', 'ESCOLA');
            // Limpar nome da escola para usar em nome de arquivo (remover caracteres especiais)
            $nomeEscolaLimpo = preg_replace('/[^a-zA-Z0-9_-]/', '_', $nomeEscola);
            $nomeEscolaLimpo = preg_replace('/_+/', '_', $nomeEscolaLimpo); // Remover múltiplos underscores
            $nomeEscolaLimpo = trim($nomeEscolaLimpo, '_');
            $nomeEscolaLimpo = substr($nomeEscolaLimpo, 0, 50); // Limitar tamanho
            
            // Data formatada
            $dataFormatada = date('Y-m-d');
            $timestamp = date('Y-m-d_H-i-s');
            
            $arquivosCriados = [];
            
            Log::info('Iniciando backup', ['tipo' => $tipo, 'timestamp' => $timestamp, 'escola' => $nomeEscola, 'diretorio' => $backupDir]);
            
            // Backup do banco de dados
            if ($tipo === 'database' || $tipo === 'completo') {
                $backupFile = $backupDir . DIRECTORY_SEPARATOR . $nomeEscolaLimpo . '_DB_' . $dataFormatada . '_' . date('H-i-s') . '.sql';
                
                $database = config('database.connections.mysql.database');
                $username = config('database.connections.mysql.username');
                $password = config('database.connections.mysql.password');
                $host = config('database.connections.mysql.host');
                $port = config('database.connections.mysql.port', 3306);
                
                $mysqldumpPath = $this->findMysqldump();
                
                if (!$mysqldumpPath) {
                    return redirect()->route('superadmin.backups.index')->withErrors(['error' => 'mysqldump não encontrado. Certifique-se de que o MySQL está instalado e no PATH.']);
                }
                
                if (strtoupper(substr(PHP_OS, 0, 3)) === 'WIN') {
                    $command = sprintf(
                        '"%s" --user=%s --password=%s --host=%s --port=%s %s > "%s"',
                        $mysqldumpPath,
                        escapeshellarg($username),
                        escapeshellarg($password),
                        escapeshellarg($host),
                        $port,
                        escapeshellarg($database),
                        $backupFile
                    );
                } else {
                    $command = sprintf(
                        '%s --user=%s --password=%s --host=%s --port=%s %s > %s',
                        $mysqldumpPath,
                        escapeshellarg($username),
                        escapeshellarg($password),
                        escapeshellarg($host),
                        $port,
                        escapeshellarg($database),
                        escapeshellarg($backupFile)
                    );
                }
                
                exec($command . ' 2>&1', $output, $returnVar);
                
                if ($returnVar !== 0 || !File::exists($backupFile) || File::size($backupFile) === 0) {
                    $errorMsg = !empty($output) ? implode("\n", $output) : 'Erro desconhecido';
                    return redirect()->route('superadmin.backups.index')->withErrors(['error' => 'Erro ao criar backup do banco: ' . $errorMsg]);
                }
                
                $arquivosCriados[] = $backupFile;
            }
            
            // Backup dos arquivos
            if ($tipo === 'files' || $tipo === 'completo') {
                $backupZip = $backupDir . DIRECTORY_SEPARATOR . $nomeEscolaLimpo . '_ARQUIVOS_' . $dataFormatada . '_' . date('H-i-s') . '.zip';
                
                if (!class_exists('ZipArchive')) {
                    return redirect()->route('superadmin.backups.index')->withErrors(['error' => 'Extensão ZipArchive não está disponível. Instale php-zip.']);
                }
                
                $zip = new ZipArchive();
                if ($zip->open($backupZip, ZipArchive::CREATE | ZipArchive::OVERWRITE) !== TRUE) {
                    return redirect()->route('superadmin.backups.index')->withErrors(['error' => 'Não foi possível criar o arquivo ZIP.']);
                }
                
                // Diretórios para fazer backup
                $diretorios = [
                    storage_path('app/public') => 'public',
                ];
                
                foreach ($diretorios as $caminho => $nomeNoZip) {
                    if (File::isDirectory($caminho)) {
                        $this->adicionarDiretorioAoZip($zip, $caminho, $nomeNoZip);
                    }
                }
                
                $zip->close();
                
                if (!File::exists($backupZip) || File::size($backupZip) === 0) {
                    Log::error('Backup de arquivos criado mas está vazio ou não existe', ['arquivo' => $backupZip]);
                    return redirect()->route('superadmin.backups.index')->withErrors(['error' => 'Erro ao criar backup dos arquivos. O arquivo ZIP está vazio ou não foi criado.']);
                }
                
                $arquivosCriados[] = $backupZip;
                Log::info('Backup de arquivos criado com sucesso', ['arquivo' => $backupZip, 'tamanho' => File::size($backupZip)]);
            }
            
            // Se for backup completo, criar um ZIP final com tudo
            if ($tipo === 'completo' && count($arquivosCriados) > 1) {
                $backupCompleto = $backupDir . DIRECTORY_SEPARATOR . $nomeEscolaLimpo . '_COMPLETO_' . $dataFormatada . '_' . date('H-i-s') . '.zip';
                $zip = new ZipArchive();
                
                if ($zip->open($backupCompleto, ZipArchive::CREATE | ZipArchive::OVERWRITE) === TRUE) {
                    foreach ($arquivosCriados as $arquivo) {
                        $zip->addFile($arquivo, basename($arquivo));
                    }
                    $zip->close();
                    
                    // Deletar arquivos individuais após criar o completo
                    foreach ($arquivosCriados as $arquivo) {
                        if (File::exists($arquivo)) {
                            File::delete($arquivo);
                        }
                    }
                    
                    $arquivosCriados = [$backupCompleto];
                }
            }
            
            // Log da ação
            Auditoria::log('BACKUP_CREATED', 'Backup', null, null, [
                'tipo' => $tipo,
                'escola' => $nomeEscola,
                'data' => $dataFormatada,
                'arquivos' => array_map('basename', $arquivosCriados)
            ]);
            
            Log::info('Backup criado com sucesso', ['tipo' => $tipo, 'arquivos' => array_map('basename', $arquivosCriados)]);
            
            // Tentar enviar para Google Drive se configurado
            $enviadoGoogleDrive = false;
            try {
                $googleDriveEnabled = Configuracao::get('google_drive_enabled', false);
                if ($googleDriveEnabled) {
                    $googleDriveService = new GoogleDriveService();
                    if ($googleDriveService->isConfigured()) {
                        foreach ($arquivosCriados as $arquivo) {
                            $resultado = $googleDriveService->uploadFile($arquivo, basename($arquivo));
                            if ($resultado['success']) {
                                $enviadoGoogleDrive = true;
                                Log::info('Backup enviado para Google Drive', ['arquivo' => basename($arquivo)]);
                            }
                        }
                    }
                }
            } catch (\Exception $e) {
                Log::warning('Erro ao enviar para Google Drive (não crítico): ' . $e->getMessage());
                // Não falha o backup se o Google Drive falhar
            }
            
            // Limpeza automática de backups muito antigos (mais de 90 dias)
            $this->limparBackupsAutomatico(90);
            
            $mensagem = 'Backup criado com sucesso!';
            if ($tipo === 'database') {
                $mensagem = 'Backup do banco de dados criado com sucesso!';
            } elseif ($tipo === 'files') {
                $mensagem = 'Backup dos arquivos criado com sucesso!';
            } elseif ($tipo === 'completo') {
                $mensagem = 'Backup completo (banco + arquivos) criado com sucesso!';
            }
            
            if ($enviadoGoogleDrive) {
                $mensagem .= ' Backup também enviado para o Google Drive!';
            }
            
            return redirect()->route('superadmin.backups.index')->with('success', $mensagem);
            
        } catch (\Exception $e) {
            Log::error('Erro ao criar backup', [
                'tipo' => $request->input('tipo', 'desconhecido'),
                'erro' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return redirect()->route('superadmin.backups.index')->withErrors(['error' => 'Erro ao criar backup: ' . $e->getMessage()]);
        }
    }

    private function limparBackupsAutomatico($dias = 90)
    {
        try {
            $backupDir = storage_path('app/backups');
            if (!File::isDirectory($backupDir)) {
                return;
            }
            
            $files = File::files($backupDir);
            $limiteData = strtotime("-{$dias} days");
            $deletados = 0;
            
            foreach ($files as $file) {
                if ($file->getMTime() < $limiteData) {
                    File::delete($file->getRealPath());
                    $deletados++;
                }
            }
            
            if ($deletados > 0) {
                Log::info("Limpeza automática: {$deletados} backup(s) antigo(s) deletado(s)");
            }
        } catch (\Exception $e) {
            Log::warning('Erro na limpeza automática de backups: ' . $e->getMessage());
        }
    }

    private function adicionarDiretorioAoZip($zip, $diretorio, $nomeNoZip)
    {
        if (!File::isDirectory($diretorio)) {
            return;
        }
        
        $iterator = new \RecursiveIteratorIterator(
            new \RecursiveDirectoryIterator($diretorio, \RecursiveDirectoryIterator::SKIP_DOTS),
            \RecursiveIteratorIterator::SELF_FIRST
        );
        
        foreach ($iterator as $item) {
            if ($item->isFile()) {
                $filePath = $item->getRealPath();
                // Normalizar separadores de diretório para usar /
                $relativePath = str_replace('\\', '/', substr($filePath, strlen($diretorio) + 1));
                $relativePath = $nomeNoZip . '/' . $relativePath;
                $zip->addFile($filePath, $relativePath);
            }
        }
    }

    private function findMysqldump()
    {
        // Tentar encontrar mysqldump no PATH
        $paths = [
            'mysqldump',
            'C:\\xampp\\mysql\\bin\\mysqldump.exe',
            'C:\\wamp\\bin\\mysql\\mysql' . $this->getMysqlVersion() . '\\bin\\mysqldump.exe',
            'C:\\Program Files\\MySQL\\MySQL Server ' . $this->getMysqlVersion() . '\\bin\\mysqldump.exe',
            'C:\\Program Files (x86)\\MySQL\\MySQL Server ' . $this->getMysqlVersion() . '\\bin\\mysqldump.exe',
        ];
        
        foreach ($paths as $path) {
            if (strtoupper(substr(PHP_OS, 0, 3)) === 'WIN') {
                // Windows: verificar se o arquivo existe
                if (file_exists($path)) {
                    return $path;
                }
            } else {
                // Linux/Unix: usar which
                $which = shell_exec("which $path 2>/dev/null");
                if ($which) {
                    return trim($which);
                }
            }
        }
        
        return null;
    }

    private function getMysqlVersion()
    {
        // Tentar detectar versão do MySQL
        try {
            $version = DB::select('SELECT VERSION() as version')[0]->version ?? '8.0';
            return substr($version, 0, 3);
        } catch (\Exception $e) {
            return '8.0'; // Versão padrão
        }
    }

    public function downloadBackup($nome)
    {
        $backupFile = storage_path('app/backups/' . $nome);
        
        if (File::exists($backupFile)) {
            Auditoria::log('BACKUP_DOWNLOADED', 'Backup', null, null, ['arquivo' => $nome]);
            return response()->download($backupFile);
        }
        
        abort(404);
    }

    public function deletarBackup($nome)
    {
        $backupFile = storage_path('app/backups/' . $nome);
        
        if (File::exists($backupFile)) {
            File::delete($backupFile);
            Auditoria::log('BACKUP_DELETED', 'Backup', null, null, ['arquivo' => $nome]);
            return redirect()->route('superadmin.backups.index')->with('success', 'Backup deletado com sucesso!');
        }
        
        abort(404);
    }

    // Configurações Google Drive
    public function configuracoesGoogleDrive()
    {
        $googleDriveEnabled = Configuracao::get('google_drive_enabled', false);
        $clientId = Configuracao::get('google_drive_client_id', '');
        $clientSecret = Configuracao::get('google_drive_client_secret', '');
        $folderId = Configuracao::get('google_drive_folder_id', '');
        $refreshToken = Configuracao::get('google_drive_refresh_token', '');
        
        $isConfigured = !empty($clientId) && !empty($clientSecret);
        $isAuthenticated = !empty($refreshToken);
        
        $folders = [];
        if ($isAuthenticated) {
            try {
                $googleDriveService = new GoogleDriveService();
                $folders = $googleDriveService->listFolders();
            } catch (\Exception $e) {
                Log::error('Erro ao listar pastas do Google Drive: ' . $e->getMessage());
            }
        }
        
        return view('superadmin.backups.google-drive', compact(
            'googleDriveEnabled', 
            'clientId', 
            'clientSecret', 
            'folderId', 
            'isConfigured', 
            'isAuthenticated',
            'folders'
        ));
    }

    public function updateConfiguracoesGoogleDrive(Request $request)
    {
        $request->validate([
            'google_drive_enabled' => 'nullable|boolean',
            'google_drive_client_id' => 'nullable|string',
            'google_drive_client_secret' => 'nullable|string',
            'google_drive_folder_id' => 'nullable|string',
        ]);

        $enabled = $request->has('google_drive_enabled') ? true : false;
        Configuracao::set('google_drive_enabled', $enabled, 'boolean', 'Habilita envio automático de backups para Google Drive');

        if ($request->filled('google_drive_client_id')) {
            Configuracao::set('google_drive_client_id', $request->google_drive_client_id, 'string', 'Client ID do Google Drive API');
        }

        if ($request->filled('google_drive_client_secret')) {
            Configuracao::set('google_drive_client_secret', $request->google_drive_client_secret, 'string', 'Client Secret do Google Drive API');
        }

        if ($request->filled('google_drive_folder_id')) {
            Configuracao::set('google_drive_folder_id', $request->google_drive_folder_id, 'string', 'ID da pasta no Google Drive onde os backups serão salvos');
        }

        Auditoria::log('UPDATE', 'Configuracao', null, null, ['configuracao' => 'google_drive']);

        return redirect()->route('superadmin.backups.google-drive')->with('success', 'Configurações do Google Drive atualizadas com sucesso!');
    }

    public function googleDriveAuth()
    {
        try {
            $googleDriveService = new GoogleDriveService();
            $authUrl = $googleDriveService->getAuthUrl();
            
            if ($authUrl) {
                return redirect($authUrl);
            }
            
            return redirect()->route('superadmin.backups.google-drive')
                ->withErrors(['error' => 'Erro ao gerar URL de autenticação. Verifique se as credenciais estão configuradas.']);
        } catch (\Exception $e) {
            return redirect()->route('superadmin.backups.google-drive')
                ->withErrors(['error' => 'Erro: ' . $e->getMessage()]);
        }
    }

    public function googleDriveCallback(Request $request)
    {
        try {
            $code = $request->input('code');
            
            if (!$code) {
                return redirect()->route('superadmin.backups.google-drive')
                    ->withErrors(['error' => 'Código de autorização não recebido.']);
            }

            $googleDriveService = new GoogleDriveService();
            $success = $googleDriveService->handleCallback($code);
            
            if ($success) {
                Auditoria::log('GOOGLE_DRIVE_AUTH', 'Configuracao', null, null, ['acao' => 'autenticacao_concluida']);
                return redirect()->route('superadmin.backups.google-drive')
                    ->with('success', 'Autenticação com Google Drive concluída com sucesso!');
            }
            
            return redirect()->route('superadmin.backups.google-drive')
                ->withErrors(['error' => 'Erro ao processar autenticação.']);
        } catch (\Exception $e) {
            Log::error('Erro no callback do Google Drive: ' . $e->getMessage());
            return redirect()->route('superadmin.backups.google-drive')
                ->withErrors(['error' => 'Erro: ' . $e->getMessage()]);
        }
    }

    // Modo de Manutenção
    public function manutencao()
    {
        $manutencaoAtiva = Configuracao::get('manutencao_ativa', false);
        $mensagemManutencao = Configuracao::get('manutencao_mensagem', 'O sistema está em manutenção. Por favor, tente novamente mais tarde.');
        $dataFimManutencao = Configuracao::get('manutencao_data_fim', null);
        
        return view('superadmin.manutencao', compact('manutencaoAtiva', 'mensagemManutencao', 'dataFimManutencao'));
    }

    public function updateManutencao(Request $request)
    {
        $request->validate([
            'manutencao_mensagem' => 'nullable|string|max:1000',
            'manutencao_data_fim' => 'nullable|date',
        ]);

        // Checkbox: se existe no request, está marcado (true), senão está desmarcado (false)
        $ativa = $request->has('manutencao_ativa');
        Configuracao::set('manutencao_ativa', $ativa, 'boolean', 'Ativa ou desativa o modo de manutenção do sistema');

        if ($request->filled('manutencao_mensagem')) {
            Configuracao::set('manutencao_mensagem', $request->manutencao_mensagem, 'string', 'Mensagem exibida na página de manutenção');
        }

        if ($request->filled('manutencao_data_fim')) {
            Configuracao::set('manutencao_data_fim', $request->manutencao_data_fim, 'string', 'Data prevista para fim da manutenção');
        } else {
            // Remover data se não fornecida
            Configuracao::where('chave', 'manutencao_data_fim')->delete();
        }

        Auditoria::log('UPDATE', 'Configuracao', null, null, [
            'configuracao' => 'manutencao',
            'ativa' => $ativa
        ]);

        $mensagem = $ativa ? 'Modo de manutenção ATIVADO. O sistema está offline para todos os usuários (exceto superadmin).' : 'Modo de manutenção DESATIVADO. O sistema está online.';
        
        return redirect()->route('superadmin.manutencao')->with('success', $mensagem);
    }
}



