<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Aluno extends Model
{
    use HasFactory, SoftDeletes;

    protected $table = 'alunos';

    protected $fillable = [
        'user_id',
        'codigo_estudante',
        'telefone',
        'endereco',
        'data_nascimento',
        'local_nascimento',
        'naturalidade',
        'distrito_natural',
        'provincia_natural',
        'num_irmaos_mais_velhos',
        'num_irmaos_mais_novos',
        'num_irmaos_mesma_idade',
        'num_documento',
        'tipo_documento',
        'foto',
        'local_emissao',
        'data_emissao',
        'nome_pai',
        'local_trabalho_pai',
        'endereco_pai_bairro',
        'endereco_pai_rua',
        'endereco_pai_numero',
        'endereco_pai_quarteirao',
        'endereco_pai_flat',
        'telefone_pai',
        'bi_pai',
        'nome_mae',
        'local_trabalho_mae',
        'endereco_mae_bairro',
        'endereco_mae_rua',
        'endereco_mae_numero',
        'endereco_mae_quarteirao',
        'endereco_mae_flat',
        'telefone_mae',
        'bi_mae',
        'genero',
        'plano_saude',
        'problema_saude_cronico',
        'alergias',
        'diagnostico_deficiencia',
        'dificuldade_motora_auditiva_visual_fala_emocional',
        'tratamento_medico',
        'medicacao',
        'vacinas_em_dia',
        'contato_emergencia',
        'outras_informacoes_saude',
        'individuo_autorizado_1_nome',
        'individuo_autorizado_1_telefone',
        'individuo_autorizado_2_nome',
        'individuo_autorizado_2_telefone',
        'individuo_autorizado_3_nome',
        'individuo_autorizado_3_telefone',
        'nome_encarregado',
        'telefone_encarregado',
        'endereco_encarregado',
        'profissao_encarregado',
        'bi_encarregado',
        'grau_parentesco',
        'pago',
        'modalidade_pagamento',
        'valor_mensalidade_personalizado',
        'status_inscricao',
        'observacoes_inscricao',
        'data_submissao',
    ];

    protected $casts = [
        'data_nascimento' => 'date',
        'data_emissao' => 'date',
        'pago' => 'boolean',
        'valor_mensalidade_personalizado' => 'decimal:2',
        'vacinas_em_dia' => 'boolean',
    ];

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function turmas()
    {
        return $this->belongsToMany(Turma::class, 'turma_aluno')
            ->withPivot('ano_letivo_id')
            ->withTimestamps()
            ->whereNull('turmas.deleted_at');
    }

    public function avaliacoes()
    {
        return $this->hasMany(Avaliacao::class);
    }

    public function trabalhos()
    {
        return $this->belongsToMany(Trabalho::class, 'trabalho_aluno')
            ->withPivot('nota', 'observacoes', 'entregue', 'data_entrega')
            ->withTimestamps();
    }

    public function financeiro()
    {
        return $this->hasMany(Financeiro::class);
    }

    public function payments()
    {
        return $this->hasMany(Payment::class);
    }

    public function invoices()
    {
        return $this->hasMany(Invoice::class);
    }

    public function getSituacaoFinanceiraAttribute()
    {
        $anoLetivoAtivo = \App\Models\AnoLetivo::getAnoLetivoAtual();
        if (!$anoLetivoAtivo) {
            return 'REGULAR';
        }

        // Data limite: 4 dias atrás (após o período de graça - dia 5 é última data sem multa)
        // Considerar apenas mensalidades vencidas há mais de 4 dias (a partir do dia 6)
        // Última data sem multa: dia 5. A partir do dia 6 começa a multa.
        // IMPORTANTE: Não considerar mensalidades futuras ou já pagas
        $dataLimite = now()->subDays(4);

        $dividas = $this->financeiro()
            ->where('ano_letivo_id', $anoLetivoAtivo->id)
            ->where('tipo', 'Mensalidade') // Apenas mensalidades
            ->whereIn('status', ['PENDENTE', 'VENCIDO']) // Apenas não pagas
            ->where('data_vencimento', '<=', $dataLimite) // Apenas vencidas há mais de 4 dias (a partir do dia 6)
            ->sum('valor');

        return $dividas > 0 ? 'EM_DÍVIDA' : 'REGULAR';
    }
}

