<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Avaliacao extends Model
{
    use HasFactory;

    protected $table = 'avaliacoes';

    protected $fillable = [
        'aluno_id',
        'disciplina_id',
        'turma_id',
        'trimestre_id',
        'ano_letivo_id',
        'modelo',
        'a_pratica_1',
        'a_pratica_2',
        'map',
        'acs_1',
        'acs_2',
        'macs',
        'at',
        'mt',
        'mfrd',
        'teste1',
        'teste2',
        'avaliacao_final',
        'media_nacional',
        'midterm_exam',
        'term_exam',
        'percent_final',
        'letra',
        'pum',
    ];

    protected $casts = [
        'a_pratica_1' => 'decimal:2',
        'a_pratica_2' => 'decimal:2',
        'map' => 'decimal:2',
        'acs_1' => 'decimal:2',
        'acs_2' => 'decimal:2',
        'macs' => 'decimal:2',
        'at' => 'decimal:2',
        'mt' => 'decimal:2',
        'mfrd' => 'decimal:2',
        'teste1' => 'decimal:2',
        'teste2' => 'decimal:2',
        'avaliacao_final' => 'decimal:2',
        'media_nacional' => 'decimal:2',
        'midterm_exam' => 'decimal:2',
        'term_exam' => 'decimal:2',
        'percent_final' => 'decimal:2',
        'pum' => 'decimal:2',
    ];

    public function aluno()
    {
        return $this->belongsTo(Aluno::class);
    }

    public function disciplina()
    {
        return $this->belongsTo(Disciplina::class);
    }

    public function turma()
    {
        return $this->belongsTo(Turma::class);
    }

    public function trimestre()
    {
        return $this->belongsTo(Trimestre::class);
    }

    public function anoLetivo()
    {
        return $this->belongsTo(AnoLetivo::class);
    }

    public function calcularMediaNacional()
    {
        if ($this->modelo !== 'NACIONAL') {
            return null;
        }

        if ($this->teste1 === null || $this->teste2 === null || $this->avaliacao_final === null) {
            return null;
        }

        $media = (($this->teste1 + $this->teste2) * 0.40) + ($this->avaliacao_final * 0.60);
        $this->media_nacional = round($media, 2);
        return $this->media_nacional;
    }

    // Cambridge removido

    /**
     * Calcula MAP (Média A.Prática)
     * MAP = (a_pratica_1 + a_pratica_2) / 2
     */
    public function calcularMAP()
    {
        if ($this->a_pratica_1 === null || $this->a_pratica_2 === null) {
            return null;
        }
        $this->map = round(($this->a_pratica_1 + $this->a_pratica_2) / 2, 2);
        return $this->map;
    }

    /**
     * Calcula MACS (Média ACS)
     * MACS = (acs_1 + acs_2) / 2
     */
    public function calcularMACS()
    {
        if ($this->acs_1 === null || $this->acs_2 === null) {
            return null;
        }
        $this->macs = round(($this->acs_1 + $this->acs_2) / 2, 2);
        return $this->macs;
    }

    /**
     * Calcula MT (Média Trimestral)
     * MT = (MAP + MACS + AT) / 3
     */
    public function calcularMT()
    {
        if ($this->map === null || $this->macs === null || $this->at === null) {
            return null;
        }
        $this->mt = round(($this->map + $this->macs + $this->at) / 3, 2);
        return $this->mt;
    }

    /**
     * Calcula MFRD (Média Final da Disciplina)
     * MFRD = (MT1 + MT2 + MT3) / 3
     * Deve ser chamado após calcular todas as médias trimestrais
     */
    public static function calcularMFRD($alunoId, $disciplinaId, $turmaId, $anoLetivoId)
    {
        $avaliacoes = self::where('aluno_id', $alunoId)
            ->where('disciplina_id', $disciplinaId)
            ->where('turma_id', $turmaId)
            ->where('ano_letivo_id', $anoLetivoId)
            ->whereNotNull('mt')
            ->get();

        if ($avaliacoes->count() < 3) {
            return null;
        }

        $somaMT = $avaliacoes->sum('mt');
        $mfrd = round($somaMT / 3, 2);

        // Atualiza todas as avaliações com a mesma MFRD
        foreach ($avaliacoes as $avaliacao) {
            $avaliacao->mfrd = $mfrd;
            $avaliacao->save();
        }

        return $mfrd;
    }

    /**
     * Verifica se o aluno está aprovado baseado no MFrD
     * Considera arredondamento por excesso: valores entre 9.5 e 9.99 são arredondados para 10
     * 
     * @param float|null $mfrd
     * @return bool
     */
    public static function isAprovado($mfrd)
    {
        if ($mfrd === null) {
            return false;
        }

        $mfrdFloat = floatval($mfrd);
        
        // Arredondar por excesso: valores entre 9.5 e 9.99 são considerados 10
        if ($mfrdFloat >= 9.5 && $mfrdFloat < 10.0) {
            $mfrdFloat = 10.0;
        }
        
        // Aprovado se MFrD >= 10
        return $mfrdFloat >= 10.0;
    }
}



