<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Funcionario extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'user_id',
        'numero_funcionario',
        'telefone',
        'endereco',
        'cargo',
        'salario_base',
        'numero_dependentes',
        'gerir_alunos',
        'gerir_professores',
        'gerir_turmas',
        'gerir_disciplinas',
        'gerir_mensalidades',
        'gerir_horarios',
        'gerir_eventos',
        'gerir_planos',
        'gerir_avaliacoes',
        'gerir_inscricoes',
        'gerir_recuperacao_senha',
        'gerir_pagamentos_professores',
    ];

    protected $casts = [
        'salario_base' => 'decimal:2',
        'numero_dependentes' => 'integer',
        'gerir_alunos' => 'boolean',
        'gerir_professores' => 'boolean',
        'gerir_turmas' => 'boolean',
        'gerir_disciplinas' => 'boolean',
        'gerir_mensalidades' => 'boolean',
        'gerir_horarios' => 'boolean',
        'gerir_eventos' => 'boolean',
        'gerir_planos' => 'boolean',
        'gerir_avaliacoes' => 'boolean',
        'gerir_inscricoes' => 'boolean',
        'gerir_recuperacao_senha' => 'boolean',
        'gerir_pagamentos_professores' => 'boolean',
    ];

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Verifica se o funcionário tem uma permissão específica
     */
    public function temPermissao($permissao)
    {
        $permissaoMap = [
            'gerir_alunos' => 'gerir_alunos',
            'gerir_professores' => 'gerir_professores',
            'gerir_turmas' => 'gerir_turmas',
            'gerir_disciplinas' => 'gerir_disciplinas',
            'gerir_mensalidades' => 'gerir_mensalidades',
            'gerir_horarios' => 'gerir_horarios',
            'gerir_eventos' => 'gerir_eventos',
            'gerir_planos' => 'gerir_planos',
            'gerir_avaliacoes' => 'gerir_avaliacoes',
            'gerir_inscricoes' => 'gerir_inscricoes',
            'gerir_recuperacao_senha' => 'gerir_recuperacao_senha',
            'gerir_pagamentos_professores' => 'gerir_pagamentos_professores',
        ];

        $campo = $permissaoMap[$permissao] ?? null;
        if (!$campo) {
            return false;
        }
        
        // Recarregar do banco para garantir dados atualizados
        $this->refresh();
        
        // Pegar o valor do atributo
        $valor = $this->getAttribute($campo);
        
        // Verificar se o valor é true (1, '1', true, 'true', ou boolean true)
        return ($valor === 1 || $valor === '1' || $valor === true || $valor === 'true' || (is_bool($valor) && $valor === true));
    }

    /**
     * Retorna todas as permissões ativas
     */
    public function getPermissoesAtivas()
    {
        $permissoes = [];
        // Lista explícita de campos de permissão (não usar fillable para evitar campos não relacionados)
        $camposPermissao = [
            'gerir_alunos',
            'gerir_professores',
            'gerir_turmas',
            'gerir_disciplinas',
            'gerir_mensalidades',
            'gerir_horarios',
            'gerir_eventos',
            'gerir_planos',
            'gerir_avaliacoes',
            'gerir_inscricoes',
            'gerir_recuperacao_senha',
            'gerir_pagamentos_professores',
        ];
        
        foreach ($camposPermissao as $campo) {
            // Pegar o valor raw do banco (antes do cast)
            $valorRaw = $this->getRawOriginal($campo);
            
            // Verificar se o valor é 1 (true) no banco
            // MySQL boolean retorna como 0 ou 1 (integer)
            if ($valorRaw === 1 || $valorRaw === '1' || $valorRaw === true) {
                $permissoes[] = $campo;
            }
        }
        
        return $permissoes;
    }
}
