<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Professor extends Model
{
    use HasFactory, SoftDeletes;

    protected $table = 'professores';

    protected $fillable = [
        'user_id',
        'numero_funcionario',
        'telefone',
        'endereco',
        'documento_identificacao',
        'data_nascimento',
        'genero',
        'nivel_ensino',
        'salario_fixo',
        'valor_hora',
        'numero_dependentes',
    ];

    protected $casts = [
        'data_nascimento' => 'date',
        'salario_fixo' => 'decimal:2',
        'valor_hora' => 'decimal:2',
    ];

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function turmas()
    {
        return $this->belongsToMany(Turma::class, 'turma_disciplina_professor')
            ->withPivot('disciplina_id', 'ano_letivo_id')
            ->withTimestamps();
    }

    public function disciplinas()
    {
        return $this->belongsToMany(Disciplina::class, 'turma_disciplina_professor')
            ->withPivot('turma_id', 'ano_letivo_id')
            ->withTimestamps();
    }

    public function horarios()
    {
        return $this->hasMany(Horario::class);
    }

    public function trabalhos()
    {
        return $this->hasMany(Trabalho::class);
    }

    public function payments()
    {
        return $this->hasMany(ProfessorPayment::class);
    }

    public function presencas()
    {
        return $this->hasMany(PresencaProfessor::class);
    }

    /**
     * Calcular salário baseado no nível de ensino
     */
    public function calcularSalario($horasTrabalhadas = null)
    {
        if ($this->nivel_ensino === 'PRIMARIO') {
            // Para primária, retorna salário fixo
            return $this->salario_fixo ?? 0;
        } elseif ($this->nivel_ensino === 'SECUNDARIO') {
            // Para secundário, calcula por horas
            if ($horasTrabalhadas === null || $horasTrabalhadas <= 0) {
                return 0;
            }
            return ($this->valor_hora ?? 0) * $horasTrabalhadas;
        } elseif ($this->nivel_ensino === 'AMBOS') {
            // Para "Ambos": salário fixo + (horas * valor_hora)
            $total = $this->salario_fixo ?? 0;
            if ($horasTrabalhadas !== null && $horasTrabalhadas > 0) {
                $total += ($this->valor_hora ?? 0) * $horasTrabalhadas;
            }
            return $total;
        }
        
        return 0;
    }
}



