<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Illuminate\Support\Facades\Hash;
use App\Models\Funcionario;

class User extends Authenticatable
{
    use HasFactory, Notifiable, SoftDeletes;

    protected $fillable = [
        'name',
        'email',
        'password',
        'tipo',
        'must_change_password',
        'is_active',
        'last_login_at',
    ];

    protected $hidden = [
        'password',
        'remember_token',
    ];

    protected $casts = [
        'email_verified_at' => 'datetime',
        'last_login_at' => 'datetime',
        'must_change_password' => 'boolean',
        'is_active' => 'boolean',
    ];

    public function setPasswordAttribute($value)
    {
        // Só faz hash se o valor ainda não estiver hasheado
        if (!empty($value) && !preg_match('/^\$2[ayb]\$\d{2}\$/', $value)) {
            $this->attributes['password'] = Hash::make($value);
        } else {
            $this->attributes['password'] = $value;
        }
    }

    public function roles()
    {
        return $this->belongsToMany(Role::class, 'user_role');
    }

    public function permissions()
    {
        return $this->belongsToMany(Permission::class, 'user_permission');
    }

    public function hasRole($role)
    {
        if (is_string($role)) {
            return $this->roles->contains('slug', $role);
        }
        return $this->roles->contains('id', $role->id);
    }

    public function hasPermission($permission)
    {
        // Verifica permissão direta
        if ($this->permissions->contains('slug', $permission)) {
            return true;
        }

        // Verifica permissão através de roles
        foreach ($this->roles as $role) {
            if ($role->permissions->contains('slug', $permission)) {
                return true;
            }
        }

        return false;
    }

    public function isSuperadmin()
    {
        return $this->tipo === 'superadmin';
    }

    public function isAdmin()
    {
        return $this->tipo === 'admin';
    }

    public function isFuncionario()
    {
        return $this->tipo === 'funcionario';
    }

    public function isProfessor()
    {
        return $this->tipo === 'professor';
    }

    public function isAluno()
    {
        return $this->tipo === 'aluno';
    }

    public function professor()
    {
        return $this->hasOne(Professor::class);
    }

    public function aluno()
    {
        return $this->hasOne(Aluno::class);
    }

    public function funcionario()
    {
        return $this->hasOne(Funcionario::class);
    }

    /**
     * Verifica se o usuário (funcionário) tem permissão para uma ação específica
     */
    public function podeGerir($recurso)
    {
        // Admin e Superadmin têm todas as permissões
        if ($this->isAdmin() || $this->isSuperadmin()) {
            return true;
        }

        // Se for funcionário, verifica permissões específicas
        if ($this->isFuncionario()) {
            // Sempre recarregar do banco para garantir dados atualizados
            $funcionario = Funcionario::where('user_id', $this->id)->first();
            
            if ($funcionario) {
                // Verificar diretamente o campo no banco usando getRawOriginal
                $campo = $recurso;
                try {
                    $valorRaw = $funcionario->getRawOriginal($campo);
                } catch (\Exception $e) {
                    $valorRaw = $funcionario->getAttribute($campo);
                }
                
                // Verificar se o valor é true (1, '1', true, 'true', ou boolean true)
                return ($valorRaw === 1 || $valorRaw === '1' || $valorRaw === true || $valorRaw === 'true' || (is_bool($valorRaw) && $valorRaw === true));
            }
        }

        return false;
    }

    public function sentMessages()
    {
        return $this->hasMany(Message::class, 'sender_id');
    }

    public function receivedMessages()
    {
        return $this->hasMany(Message::class, 'receiver_id');
    }

    public function notifications()
    {
        return $this->hasMany(Notification::class);
    }

    public function unreadNotifications()
    {
        return $this->hasMany(Notification::class)->where('read', false);
    }

    public function unreadNotificationsCount()
    {
        return Notification::where('user_id', $this->id)->where('read', false)->count();
    }
}



