<?php

namespace App\Repositories;

use App\Models\Aluno;
use App\Services\CodigoEstudanteService;
use App\Models\User;
use Illuminate\Support\Facades\Hash;

class AlunoRepository
{
    public function create(array $dados, $anoIngresso = null): Aluno
    {
        // Gerar email automático se não fornecido
        $email = $dados['email'] ?? null;
        if (!$email) {
            // Remover acentos e caracteres especiais do nome
            $nomeBase = $dados['name'];
            $nomeBase = mb_strtolower($nomeBase, 'UTF-8');
            $nomeBase = preg_replace('/[^a-z0-9\s]/', '', $nomeBase);
            $nomeBase = preg_replace('/\s+/', '', $nomeBase);
            $nomeBase = substr($nomeBase, 0, 20); // Limitar tamanho
            
            if (empty($nomeBase)) {
                $nomeBase = 'aluno';
            }
            
            $email = $nomeBase . '_' . time() . '@aluno.sge.pt';
            
            // Garantir que o email seja único
            $contador = 1;
            while (User::where('email', $email)->exists()) {
                $email = $nomeBase . '_' . time() . '_' . $contador . '@aluno.sge.pt';
                $contador++;
            }
        }
        
        // Cria o usuário primeiro
        $userData = [
            'name' => $dados['name'],
            'email' => $email,
            'password' => Hash::make($dados['password'] ?? date('Y')),
            'tipo' => 'aluno',
            'must_change_password' => true,
            'is_active' => true,
        ];

        $user = User::create($userData);

        // Gera código de estudante
        $codigoEstudante = CodigoEstudanteService::gerar($anoIngresso);

        // Cria o aluno com todos os campos
        $alunoData = array_merge([
            'user_id' => $user->id,
            'codigo_estudante' => $codigoEstudante,
            'pago' => $dados['pago'] ?? true,
        ], array_filter($dados, function($key) {
            // Incluir apenas campos que existem na tabela alunos
            return in_array($key, [
                'telefone', 'endereco', 'data_nascimento', 'local_nascimento',
                'naturalidade', 'distrito_natural', 'provincia_natural',
                'num_irmaos_mais_velhos', 'num_irmaos_mais_novos', 'num_irmaos_mesma_idade',
                'num_documento', 'tipo_documento', 'foto', 'local_emissao', 'data_emissao',
                'nome_pai', 'local_trabalho_pai', 'endereco_pai_bairro', 'endereco_pai_rua',
                'endereco_pai_numero', 'endereco_pai_quarteirao', 'endereco_pai_flat',
                'telefone_pai', 'bi_pai', 'nome_mae', 'local_trabalho_mae',
                'endereco_mae_bairro', 'endereco_mae_rua', 'endereco_mae_numero',
                'endereco_mae_quarteirao', 'endereco_mae_flat', 'telefone_mae', 'bi_mae',
                'genero', 'plano_saude', 'problema_saude_cronico', 'alergias',
                'diagnostico_deficiencia', 'dificuldade_motora_auditiva_visual_fala_emocional',
                'tratamento_medico', 'medicacao', 'vacinas_em_dia', 'contato_emergencia',
                'outras_informacoes_saude', 'individuo_autorizado_1_nome',
                'individuo_autorizado_1_telefone', 'individuo_autorizado_2_nome',
                'individuo_autorizado_2_telefone', 'individuo_autorizado_3_nome',
                'individuo_autorizado_3_telefone', 'nome_encarregado', 'telefone_encarregado',
                'endereco_encarregado', 'profissao_encarregado', 'bi_encarregado',
                'grau_parentesco', 'modalidade_pagamento', 'valor_mensalidade_personalizado'
            ]);
        }, ARRAY_FILTER_USE_KEY));

        return Aluno::create($alunoData);
    }

    public function update(Aluno $aluno, array $dados): Aluno
    {
        // Separar dados do aluno dos dados do usuário
        $userData = [];
        if (isset($dados['name'])) {
            $userData['name'] = $dados['name'];
            unset($dados['name']);
        }
        if (isset($dados['email'])) {
            $userData['email'] = $dados['email'];
            unset($dados['email']);
        }
        
        // Atualizar dados do aluno
        $aluno->update($dados);

        // Atualizar dados do usuário se necessário
        if (!empty($userData) && $aluno->user) {
            $aluno->user->update($userData);
        }

        return $aluno->fresh();
    }

    public function findByCodigoEstudante(string $codigo): ?Aluno
    {
        return Aluno::where('codigo_estudante', $codigo)->first();
    }

    public function getAll()
    {
        return Aluno::with('user')->get();
    }
}

