<?php

namespace App\Services;

use App\Models\Professor;
use Illuminate\Support\Facades\DB;

class CodigoProfessorService
{
    /**
     * Gera um código de professor único no formato PROF00001
     * PROF = prefixo fixo
     * 00001 = sequencial automático
     */
    public static function gerar(): string
    {
        // Usar lock para evitar condições de corrida
        return DB::transaction(function () {
            // Incluir soft-deleted: a unique key vale para todos os registros.
            // Busca o maior número existente usando CAST para ordenar numericamente
            $ultimoProfessor = Professor::withTrashed()
                ->where('numero_funcionario', 'like', 'PROF%')
                ->orderByRaw("CAST(SUBSTRING(numero_funcionario, 5) AS UNSIGNED) DESC")
                ->lockForUpdate()
                ->first();

            if ($ultimoProfessor && $ultimoProfessor->numero_funcionario) {
                $ultimoNumero = (int) substr($ultimoProfessor->numero_funcionario, 4);
                $novoNumero = $ultimoNumero + 1;
            } else {
                $novoNumero = 1;
            }

            $novoCodigo = 'PROF' . str_pad($novoNumero, 5, '0', STR_PAD_LEFT);
            
            // Verificar se o código já existe (inclui soft-deleted)
            while (Professor::withTrashed()->where('numero_funcionario', $novoCodigo)->exists()) {
                $novoNumero++;
                $novoCodigo = 'PROF' . str_pad($novoNumero, 5, '0', STR_PAD_LEFT);
            }
            
            return $novoCodigo;
        });
    }

    /**
     * Valida o formato do código de professor
     */
    public static function validar($codigo): bool
    {
        return preg_match('/^PROF\d{5}$/', $codigo) === 1;
    }
}
