<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\User;
use App\Models\Aluno;
use App\Models\Turma;
use App\Models\AnoLetivo;
use App\Models\Disciplina;
use App\Models\Trimestre;
use App\Models\Avaliacao;
use App\Services\CodigoEstudanteService;
use App\Services\AvaliacaoService;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\App;

class AdicionarAlunos2026Seeder extends Seeder
{
    public function run(): void
    {
        $this->command->info('=== ADICIONANDO ALUNOS PARA 2026 ===');
        
        // Buscar ou criar ano letivo 2026
        $anoLetivo2026 = AnoLetivo::firstOrCreate(
            ['ano' => '2026/2027'],
            [
                'data_inicio' => '2026-01-01',
                'data_fim' => '2027-12-31',
                'ativo' => false, // Não ativar automaticamente
            ]
        );
        
        $this->command->info("Ano letivo 2026/2027: ID {$anoLetivo2026->id}");
        
        // Criar trimestres para 2026 se não existirem
        $trimestres2026 = Trimestre::where('ano_letivo_id', $anoLetivo2026->id)->get();
        if ($trimestres2026->isEmpty()) {
            $this->command->info('Criando trimestres para 2026...');
            Trimestre::create([
                'nome' => '1º Trimestre',
                'numero' => 1,
                'ano_letivo_id' => $anoLetivo2026->id,
                'data_inicio' => '2026-01-01',
                'data_fim' => '2026-04-30',
                'ativo' => false,
            ]);
            Trimestre::create([
                'nome' => '2º Trimestre',
                'numero' => 2,
                'ano_letivo_id' => $anoLetivo2026->id,
                'data_inicio' => '2026-05-01',
                'data_fim' => '2026-08-31',
                'ativo' => false,
            ]);
            Trimestre::create([
                'nome' => '3º Trimestre',
                'numero' => 3,
                'ano_letivo_id' => $anoLetivo2026->id,
                'data_inicio' => '2026-09-01',
                'data_fim' => '2026-12-31',
                'ativo' => false,
            ]);
            $trimestres2026 = Trimestre::where('ano_letivo_id', $anoLetivo2026->id)->get();
        }
        
        // Buscar turma 1ª Classe - A do ano 2026
        $turma1A = Turma::where('codigo', '1ª Classe')
            ->where('nome', 'A')
            ->where('ano_letivo_id', $anoLetivo2026->id)
            ->first();
        
        if (!$turma1A) {
            $this->command->info('Criando turma 1ª Classe - A para 2026...');
            $turma1A = Turma::create([
                'codigo' => '1ª Classe',
                'nome' => 'A',
                'ano_letivo_id' => $anoLetivo2026->id,
                'capacidade_maxima' => 30,
                'ativa' => true,
            ]);
        }
        
        // Verificar se já existem disciplinas atribuídas à turma
        $disciplinasTurma = DB::table('turma_disciplina')
            ->where('turma_id', $turma1A->id)
            ->where('ano_letivo_id', $anoLetivo2026->id)
            ->pluck('disciplina_id')
            ->toArray();
        
        if (empty($disciplinasTurma)) {
            $this->command->info('Atribuindo disciplinas à turma 1ª Classe - A...');
            $disciplinas = Disciplina::where('ativa', true)->take(5)->get();
            foreach ($disciplinas as $disciplina) {
                DB::table('turma_disciplina')->insert([
                    'turma_id' => $turma1A->id,
                    'disciplina_id' => $disciplina->id,
                    'ano_letivo_id' => $anoLetivo2026->id,
                    'created_at' => now(),
                    'updated_at' => now(),
                ]);
            }
            $disciplinasTurma = DB::table('turma_disciplina')
                ->where('turma_id', $turma1A->id)
                ->where('ano_letivo_id', $anoLetivo2026->id)
                ->pluck('disciplina_id')
                ->toArray();
        }
        
        // Buscar alunos que entraram em 2025 (código inicia com 2025)
        $alunos2025 = Aluno::where('codigo_estudante', 'like', '2025%')->get();
        
        $this->command->info("Encontrados " . $alunos2025->count() . " alunos que entraram em 2025");
        
        // Alocar alunos de 2025 para a turma 1ª Classe - A de 2026
        $alunosAlocados = 0;
        foreach ($alunos2025->take(15) as $aluno) {
            // Verificar se já está alocado nesta turma e ano
            $jaAlocado = DB::table('turma_aluno')
                ->where('aluno_id', $aluno->id)
                ->where('turma_id', $turma1A->id)
                ->where('ano_letivo_id', $anoLetivo2026->id)
                ->exists();
            
            if (!$jaAlocado) {
                // Alocar aluno à turma mantendo o código original
                $aluno->turmas()->attach($turma1A->id, ['ano_letivo_id' => $anoLetivo2026->id]);
                $alunosAlocados++;
                
                $this->command->info("Aluno {$aluno->codigo_estudante} ({$aluno->user->name}) alocado à 1ª Classe - A de 2026");
            }
        }
        
        // Criar novos alunos para 2026 se necessário
        $totalAlunosTurma = DB::table('turma_aluno')
            ->where('turma_id', $turma1A->id)
            ->where('ano_letivo_id', $anoLetivo2026->id)
            ->count();
        
        if ($totalAlunosTurma < 20) {
            $this->command->info("Criando novos alunos para 2026...");
            $novosAlunos = $this->criarNovosAlunos2026($turma1A, $anoLetivo2026, 20 - $totalAlunosTurma);
            $alunosAlocados += $novosAlunos;
        }
        
        // Criar avaliações para TODOS os alunos da 1ª Classe - A de 2026 com aprovação garantida
        $this->command->info('Criando avaliações com aprovação garantida...');
        $this->criarAvaliacoesAprovadas($turma1A, $anoLetivo2026, $trimestres2026, $disciplinasTurma);
        
        $this->command->info("=== CONCLUÍDO: {$alunosAlocados} alunos alocados à 1ª Classe - A de 2026 ===");
    }
    
    private function criarNovosAlunos2026($turma, $anoLetivo, $quantidade)
    {
        $nomes = [
            'Ana Sofia', 'Beatriz Costa', 'Carolina Silva', 'Diana Mendes', 'Eva Fernandes',
            'Filipa Alves', 'Gabriela Sousa', 'Helena Gomes', 'Inês Ribeiro', 'Joana Pinto',
            'André Santos', 'Bruno Oliveira', 'Carlos Pereira', 'Daniel Rodrigues', 'Eduardo Martins',
            'Filipe Ferreira', 'Gonçalo Lopes', 'Hugo Carvalho', 'Ivo Correia', 'João Moreira'
        ];
        
        $generos = ['F', 'F', 'F', 'F', 'F', 'F', 'F', 'F', 'F', 'F', 'M', 'M', 'M', 'M', 'M', 'M', 'M', 'M', 'M', 'M'];
        
        $anoIngresso = 2026;
        $index = 1;
        $timestamp = time();
        $criados = 0;
        
        foreach ($nomes as $key => $nome) {
            if ($criados >= $quantidade) break;
            
            $email = strtolower(str_replace(' ', '', $nome)) . $timestamp . $index . '@aluno2026.sge.pt';
            
            if (User::where('email', $email)->exists()) {
                $index++;
                continue;
            }
            
            $user = User::create([
                'name' => $nome,
                'email' => $email,
                'password' => Hash::make('2026'),
                'tipo' => 'aluno',
                'must_change_password' => true,
                'is_active' => true,
            ]);
            
            // Gerar código para 2026
            $codigoEstudante = CodigoEstudanteService::gerar($anoIngresso);
            
            $aluno = Aluno::create([
                'user_id' => $user->id,
                'codigo_estudante' => $codigoEstudante,
                'telefone' => '+258 8' . rand(1000000, 9999999),
                'endereco' => 'Endereço ' . $index . ', Maputo',
                'data_nascimento' => now()->subYears(rand(10, 18))->subDays(rand(0, 365)),
                'genero' => $generos[$key],
                'nome_encarregado' => 'Encarregado de ' . $nome,
                'telefone_encarregado' => '+258 8' . rand(1000000, 9999999),
            ]);
            
            // Alocar à turma 1ª Classe - A de 2026
            $aluno->turmas()->attach($turma->id, ['ano_letivo_id' => $anoLetivo->id]);
            
            $this->command->info("Novo aluno criado: {$codigoEstudante} ({$nome})");
            
            $index++;
            $criados++;
        }
        
        return $criados;
    }
    
    private function criarAvaliacoesAprovadas($turma, $anoLetivo, $trimestres, $disciplinasTurma)
    {
        $alunosTurma = $turma->alunos()->wherePivot('ano_letivo_id', $anoLetivo->id)->get();
        
        if ($alunosTurma->isEmpty() || empty($disciplinasTurma)) {
            $this->command->warn('Nenhum aluno ou disciplina encontrado para criar avaliações.');
            return;
        }
        
        $avaliacoesCriadas = 0;
        
        foreach ($alunosTurma as $aluno) {
            foreach ($disciplinasTurma as $disciplinaId) {
                foreach ($trimestres as $trimestre) {
                    // Verificar se já existe avaliação
                    $existe = Avaliacao::where('aluno_id', $aluno->id)
                        ->where('disciplina_id', $disciplinaId)
                        ->where('trimestre_id', $trimestre->id)
                        ->where('ano_letivo_id', $anoLetivo->id)
                        ->exists();
                    
                    if ($existe) continue;
                    
                    // Notas entre 10-20 para garantir aprovação
                    $a_pratica_1 = rand(100, 200) / 10; // 10.0 a 20.0
                    $a_pratica_2 = rand(100, 200) / 10;
                    $acs_1 = rand(100, 200) / 10;
                    $acs_2 = rand(100, 200) / 10;
                    $at = rand(100, 200) / 10;
                    
                    $dados = [
                        'aluno_id' => $aluno->id,
                        'disciplina_id' => $disciplinaId,
                        'turma_id' => $turma->id,
                        'trimestre_id' => $trimestre->id,
                        'ano_letivo_id' => $anoLetivo->id,
                        'modelo' => 'NACIONAL',
                        'a_pratica_1' => $a_pratica_1,
                        'a_pratica_2' => $a_pratica_2,
                        'acs_1' => $acs_1,
                        'acs_2' => $acs_2,
                        'at' => $at,
                    ];
                    
                    $avaliacaoService = App::make(AvaliacaoService::class);
                    $avaliacaoService->salvarAvaliacao($dados);
                    $avaliacoesCriadas++;
                }
            }
        }
        
        $this->command->info("Criadas {$avaliacoesCriadas} avaliações com aprovação garantida.");
    }
}

