<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\Avaliacao;
use App\Models\Aluno;
use App\Models\Turma;
use App\Models\Disciplina;
use App\Models\Trimestre;
use App\Models\AnoLetivo;
use Illuminate\Support\Facades\DB;

class AvaliacaoSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        $anoLetivo = AnoLetivo::where('ativo', true)->first();
        
        if (!$anoLetivo) {
            $this->command->warn('Nenhum ano letivo ativo encontrado. Execute o AnoLetivoSeeder primeiro.');
            return;
        }

        $trimestres = Trimestre::where('ano_letivo_id', $anoLetivo->id)
            ->orderBy('numero')
            ->get();

        if ($trimestres->isEmpty()) {
            $this->command->warn('Nenhum trimestre encontrado para o ano letivo ativo.');
            return;
        }

        $turmas = Turma::where('ano_letivo_id', $anoLetivo->id)
            ->where('ativa', true)
            ->get();

        if ($turmas->isEmpty()) {
            $this->command->warn('Nenhuma turma encontrada para o ano letivo ativo.');
            return;
        }

        $disciplinas = Disciplina::where('ativa', true)->get();

        if ($disciplinas->isEmpty()) {
            $this->command->warn('Nenhuma disciplina ativa encontrada.');
            return;
        }

        $this->command->info('Iniciando população de avaliações...');
        $totalAvaliacoes = 0;

        foreach ($turmas as $turma) {
            // Buscar alunos da turma no ano letivo
            $alunos = $turma->alunos()
                ->wherePivot('ano_letivo_id', $anoLetivo->id)
                ->get();

            if ($alunos->isEmpty()) {
                $this->command->warn("Turma {$turma->nome} não tem alunos alocados.");
                continue;
            }

            // Buscar disciplinas da turma
            $disciplinasTurma = DB::table('turma_disciplina_professor')
                ->where('turma_id', $turma->id)
                ->where('ano_letivo_id', $anoLetivo->id)
                ->pluck('disciplina_id')
                ->toArray();

            if (empty($disciplinasTurma)) {
                $this->command->warn("Turma {$turma->nome} não tem disciplinas alocadas.");
                continue;
            }

            foreach ($alunos as $aluno) {
                foreach ($disciplinasTurma as $disciplinaId) {
                    $disciplina = Disciplina::find($disciplinaId);
                    
                    if (!$disciplina) {
                        continue;
                    }

                    // Criar avaliações para cada trimestre
                    foreach ($trimestres as $trimestre) {
                        // Verificar se já existe avaliação
                        $avaliacaoExistente = Avaliacao::where('aluno_id', $aluno->id)
                            ->where('disciplina_id', $disciplina->id)
                            ->where('turma_id', $turma->id)
                            ->where('trimestre_id', $trimestre->id)
                            ->where('ano_letivo_id', $anoLetivo->id)
                            ->first();

                        if ($avaliacaoExistente) {
                            continue; // Pula se já existe
                        }

                        // Gerar notas aleatórias mas realistas (entre 7 e 20)
                        // Gerar notas: reprovados (0-9.9) ou aprovados (10-20)
                        // Mais da metade deve passar (aproximadamente 65% aprovados, 35% reprovados)
                        $isReprovado = rand(1, 100) <= 35; // 35% de chance de reprovar
                        if ($isReprovado) {
                            $aPratica1 = round(rand(0, 99) / 10, 1); // 0.0 a 9.9
                            $aPratica2 = round(rand(0, 99) / 10, 1);
                            $acs1 = round(rand(0, 99) / 10, 1);
                            $acs2 = round(rand(0, 99) / 10, 1);
                            $at = round(rand(0, 99) / 10, 1);
                        } else {
                            $aPratica1 = round(rand(100, 200) / 10, 1); // 10.0 a 20.0
                            $aPratica2 = round(rand(100, 200) / 10, 1);
                            $acs1 = round(rand(100, 200) / 10, 1);
                            $acs2 = round(rand(100, 200) / 10, 1);
                            $at = round(rand(100, 200) / 10, 1);
                        }

                        // Calcular MAP
                        $map = round(($aPratica1 + $aPratica2) / 2, 2);

                        // Calcular MACS
                        $macs = round(($acs1 + $acs2) / 2, 2);

                        // Calcular MT
                        $mt = round(($map + $macs + $at) / 3, 2);

                        // Criar avaliação
                        $avaliacao = Avaliacao::create([
                            'aluno_id' => $aluno->id,
                            'disciplina_id' => $disciplina->id,
                            'turma_id' => $turma->id,
                            'trimestre_id' => $trimestre->id,
                            'ano_letivo_id' => $anoLetivo->id,
                            'modelo' => 'NACIONAL',
                            'a_pratica_1' => $aPratica1,
                            'a_pratica_2' => $aPratica2,
                            'map' => $map,
                            'acs_1' => $acs1,
                            'acs_2' => $acs2,
                            'macs' => $macs,
                            'at' => $at,
                            'mt' => $mt,
                        ]);

                        $totalAvaliacoes++;
                    }
                }
            }
        }

        // Calcular MFRD para todos os alunos/disciplinas/turmas
        $this->command->info('Calculando MFRD (Média Final da Disciplina)...');
        
        foreach ($turmas as $turma) {
            $alunos = $turma->alunos()
                ->wherePivot('ano_letivo_id', $anoLetivo->id)
                ->get();

            $disciplinasTurma = DB::table('turma_disciplina_professor')
                ->where('turma_id', $turma->id)
                ->where('ano_letivo_id', $anoLetivo->id)
                ->pluck('disciplina_id')
                ->toArray();

            foreach ($alunos as $aluno) {
                foreach ($disciplinasTurma as $disciplinaId) {
                    Avaliacao::calcularMFRD(
                        $aluno->id,
                        $disciplinaId,
                        $turma->id,
                        $anoLetivo->id
                    );
                }
            }
        }

        $this->command->info("✓ {$totalAvaliacoes} avaliações criadas com sucesso!");
    }
}
