<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\AnoLetivo;
use App\Models\Turma;
use Illuminate\Support\Facades\DB;

class CorrigirAnoLetivo2026Seeder extends Seeder
{
    public function run(): void
    {
        $this->command->info('=== CORRIGINDO ANO LETIVO PARA 2026 ===');
        
        // Buscar ou criar ano letivo 2026
        $anoLetivo2026 = AnoLetivo::where(function($q) {
            $q->where('ano', 'like', '%2026%')
              ->orWhere('ano', '2026');
        })->first();
        
        if (!$anoLetivo2026) {
            $this->command->info('Criando ano letivo 2026...');
            $anoLetivo2026 = AnoLetivo::create([
                'ano' => '2026',
                'data_inicio' => '2026-01-01',
                'data_fim' => '2026-12-31',
                'ativo' => true,
            ]);
            $this->command->info("Ano letivo 2026 criado (ID: {$anoLetivo2026->id})");
        } else {
            // Atualizar o valor do campo "ano" para garantir que seja 2026
            $anoRaw = $anoLetivo2026->getRawOriginal('ano');
            // Se contém 2026 mas não é exatamente "2026", atualizar
            if ($anoRaw !== '2026' && strpos($anoRaw, '2026') !== false) {
                $this->command->info("Atualizando ano letivo ID {$anoLetivo2026->id} de '{$anoRaw}' para '2026'");
                // Usar update direto para evitar o accessor
                DB::table('ano_letivo')
                    ->where('id', $anoLetivo2026->id)
                    ->update(['ano' => '2026']);
                $anoLetivo2026->refresh();
            } elseif ($anoRaw !== '2026' && strpos($anoRaw, '2026') === false) {
                // Se não contém 2026, atualizar também
                $this->command->info("Atualizando ano letivo ID {$anoLetivo2026->id} de '{$anoRaw}' para '2026'");
                DB::table('ano_letivo')
                    ->where('id', $anoLetivo2026->id)
                    ->update(['ano' => '2026']);
                $anoLetivo2026->refresh();
            }
            // Garantir que 2026 está ativo
            $anoLetivo2026->ativo = true;
            $anoLetivo2026->save();
            $this->command->info("Ano letivo 2026 configurado (ID: {$anoLetivo2026->id}, ano: {$anoLetivo2026->getRawOriginal('ano')})");
        }
        
        // Desativar outros anos letivos (exceto 2026)
        $outrosAnos = AnoLetivo::where('id', '!=', $anoLetivo2026->id)->get();
        foreach ($outrosAnos as $ano) {
            $ano->ativo = false;
            $ano->save();
        }
        $this->command->info("Outros anos letivos desativados: {$outrosAnos->count()}");
        
        // Atualizar turmas que não são do ano 2026
        // Primeiro, apagar turmas duplicadas (mesmo código e nome já existente em 2026)
        $turmas2026 = Turma::where('ano_letivo_id', $anoLetivo2026->id)->get();
        $turmasOutrosAnos = Turma::where('ano_letivo_id', '!=', $anoLetivo2026->id)
            ->orWhereNull('ano_letivo_id')
            ->get();
        
        $turmasApagadas = 0;
        foreach ($turmasOutrosAnos as $turma) {
            // Verificar se já existe uma turma com mesmo código e nome em 2026
            $existe = $turmas2026->first(function($t) use ($turma) {
                return $t->codigo === $turma->codigo && $t->nome === $turma->nome;
            });
            
            if ($existe) {
                // Se existe, apagar a turma do outro ano
                $turma->delete();
                $turmasApagadas++;
            } else {
                // Se não existe, atualizar para 2026
                $turma->ano_letivo_id = $anoLetivo2026->id;
                $turma->save();
            }
        }
        
        $this->command->info("Turmas apagadas (duplicadas): {$turmasApagadas}");
        $this->command->info("Turmas atualizadas para 2026: " . ($turmasOutrosAnos->count() - $turmasApagadas));
        
        // Atualizar relações turma_aluno
        $relacoesAtualizadas = DB::table('turma_aluno')
            ->where('ano_letivo_id', '!=', $anoLetivo2026->id)
            ->orWhereNull('ano_letivo_id')
            ->update(['ano_letivo_id' => $anoLetivo2026->id]);
        
        $this->command->info("Relações turma_aluno atualizadas: {$relacoesAtualizadas}");
        
        // Atualizar relações turma_disciplina
        $disciplinasAtualizadas = DB::table('turma_disciplina')
            ->where('ano_letivo_id', '!=', $anoLetivo2026->id)
            ->orWhereNull('ano_letivo_id')
            ->update(['ano_letivo_id' => $anoLetivo2026->id]);
        
        $this->command->info("Relações turma_disciplina atualizadas: {$disciplinasAtualizadas}");
        
        // Atualizar relações turma_disciplina_professor
        $professoresAtualizados = DB::table('turma_disciplina_professor')
            ->where('ano_letivo_id', '!=', $anoLetivo2026->id)
            ->orWhereNull('ano_letivo_id')
            ->update(['ano_letivo_id' => $anoLetivo2026->id]);
        
        $this->command->info("Relações turma_disciplina_professor atualizadas: {$professoresAtualizados}");
        
        $this->command->info("=== CONCLUÍDO: Sistema configurado para usar apenas 2026 ===");
    }
}
