<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\User;
use App\Models\Aluno;
use App\Models\Professor;
use App\Models\Turma;
use App\Models\Disciplina;
use App\Models\AnoLetivo;
use App\Models\Financeiro;
use App\Models\Payment;
use App\Models\Invoice;
use App\Models\Horario;
use App\Models\Avaliacao;
use App\Models\Trimestre;
use App\Services\CodigoEstudanteService;
use App\Services\CodigoProfessorService;
use App\Services\FinanceiroService;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\DB;

class CriarDadosCompletos2026Seeder extends Seeder
{
    private $anoLetivo;
    private $turmas;
    private $disciplinas;
    private $professores;
    private $financeiroService;
    
    public function run(): void
    {
        $this->command->info('=== CRIANDO DADOS COMPLETOS PARA 2026 ===');
        
        $this->financeiroService = app(\App\Services\FinanceiroService::class);
        
        // 1. Obter/Criar ano letivo 2026
        $this->anoLetivo = AnoLetivo::getAnoLetivoAtual();
        if (!$this->anoLetivo) {
            $this->anoLetivo = AnoLetivo::firstOrCreate(
                ['ano' => '2026'],
                [
                    'data_inicio' => '2026-01-01',
                    'data_fim' => '2026-12-31',
                    'ativo' => true,
                ]
            );
        }
        $this->command->info("Usando ano letivo: {$this->anoLetivo->ano} (ID: {$this->anoLetivo->id})");
        
        // 2. Criar Disciplinas
        $this->command->info('=== CRIANDO DISCIPLINAS ===');
        $this->disciplinas = $this->criarDisciplinas();
        $this->command->info("Criadas " . $this->disciplinas->count() . " disciplinas");
        
        // 3. Criar Turmas
        $this->command->info('=== CRIANDO TURMAS ===');
        $this->turmas = $this->criarTurmas();
        $this->command->info("Criadas " . $this->turmas->count() . " turmas");
        
        // 4. Criar Professores
        $this->command->info('=== CRIANDO PROFESSORES ===');
        $this->professores = $this->criarProfessores();
        $this->command->info("Criados " . $this->professores->count() . " professores");
        
        // 5. Alocar disciplinas às turmas
        $this->command->info('=== ALOCANDO DISCIPLINAS ÀS TURMAS ===');
        $this->alocarDisciplinasTurmas();
        
        // 6. Alocar professores às turmas-disciplinas
        $this->command->info('=== ALOCANDO PROFESSORES ÀS TURMAS-DISCIPLINAS ===');
        $this->alocarProfessoresTurmasDisciplinas();
        
        // 7. Criar 400 Alunos e alocá-los às turmas
        $this->command->info('=== CRIANDO 400 ALUNOS ===');
        $alunos = $this->criarAlunos(400);
        $this->command->info("Criados " . $alunos->count() . " alunos");
        
        // 8. Criar pagamentos do primeiro mês (janeiro)
        $this->command->info('=== CRIANDO PAGAMENTOS DE JANEIRO ===');
        $this->criarPagamentosJaneiro($alunos);
        
        // 9. Criar Trimestres
        $this->command->info('=== CRIANDO TRIMESTRES ===');
        $trimestre1 = $this->criarTrimestre(1);
        $trimestre2 = $this->criarTrimestre(2);
        $trimestre3 = $this->criarTrimestre(3);
        
        // 10. Criar Horários
        $this->command->info('=== CRIANDO HORÁRIOS ===');
        $this->criarHorarios();
        
        // 11. Criar Eventos
        $this->command->info('=== CRIANDO EVENTOS ===');
        $this->criarEventos();
        
        // 12. Criar Notas de Todos os Trimestres
        $this->command->info('=== CRIANDO NOTAS DO PRIMEIRO TRIMESTRE ===');
        $this->criarNotasTrimestre($trimestre1, $alunos);
        
        $this->command->info('=== CRIANDO NOTAS DO SEGUNDO TRIMESTRE ===');
        $this->criarNotasTrimestre($trimestre2, $alunos);
        
        $this->command->info('=== CRIANDO NOTAS DO TERCEIRO TRIMESTRE ===');
        $this->criarNotasTrimestre($trimestre3, $alunos);
        
        $this->command->info('=== CONCLUÍDO COM SUCESSO ===');
    }
    
    private function criarDisciplinas()
    {
        // Primeiro, buscar todas as disciplinas existentes
        $disciplinasExistentes = Disciplina::all()->keyBy(function($d) {
            return $d->codigo . '_' . $d->nivel_ensino;
        });
        
        $disciplinas = collect();
        
        // Disciplinas do Ensino Primário
        $disciplinasPrimario = [
            ['nome' => 'Língua Portuguesa', 'codigo' => 'LP', 'carga_horaria_semanal' => 300],
            ['nome' => 'Matemática', 'codigo' => 'MAT', 'carga_horaria_semanal' => 300],
            ['nome' => 'Estudo do Meio', 'codigo' => 'EM', 'carga_horaria_semanal' => 180],
            ['nome' => 'Educação Física', 'codigo' => 'EF', 'carga_horaria_semanal' => 120],
            ['nome' => 'Educação Visual', 'codigo' => 'EV', 'carga_horaria_semanal' => 120],
            ['nome' => 'Educação Musical', 'codigo' => 'MUS', 'carga_horaria_semanal' => 120],
            ['nome' => 'Educação Moral e Cívica', 'codigo' => 'EMC', 'carga_horaria_semanal' => 90],
            ['nome' => 'Língua Inglesa', 'codigo' => 'INGP', 'carga_horaria_semanal' => 120],
            ['nome' => 'Educação Artística', 'codigo' => 'EA', 'carga_horaria_semanal' => 90],
        ];
        
        // Disciplinas do Ensino Secundário
        $disciplinasSecundario = [
            ['nome' => 'Língua Portuguesa', 'codigo' => 'LPS', 'carga_horaria_semanal' => 240],
            ['nome' => 'Matemática', 'codigo' => 'MATS', 'carga_horaria_semanal' => 240],
            ['nome' => 'Física', 'codigo' => 'FIS', 'carga_horaria_semanal' => 180],
            ['nome' => 'Química', 'codigo' => 'QUI', 'carga_horaria_semanal' => 180],
            ['nome' => 'Biologia', 'codigo' => 'BIO', 'carga_horaria_semanal' => 180],
            ['nome' => 'História', 'codigo' => 'HIS', 'carga_horaria_semanal' => 180],
            ['nome' => 'Geografia', 'codigo' => 'GEO', 'carga_horaria_semanal' => 180],
            ['nome' => 'Inglês', 'codigo' => 'ING', 'carga_horaria_semanal' => 180],
            ['nome' => 'Educação Física', 'codigo' => 'EFS', 'carga_horaria_semanal' => 120],
            ['nome' => 'Educação Visual', 'codigo' => 'EVS', 'carga_horaria_semanal' => 120],
            ['nome' => 'Filosofia', 'codigo' => 'FIL', 'carga_horaria_semanal' => 120],
            ['nome' => 'Francês', 'codigo' => 'FRA', 'carga_horaria_semanal' => 120],
            ['nome' => 'Desenho Técnico', 'codigo' => 'DT', 'carga_horaria_semanal' => 120],
            ['nome' => 'TIC', 'codigo' => 'TIC', 'carga_horaria_semanal' => 120],
            ['nome' => 'Economia', 'codigo' => 'ECO', 'carga_horaria_semanal' => 120],
        ];
        
        foreach ($disciplinasPrimario as $disc) {
            $key = $disc['codigo'] . '_PRIMARIO';
            $disciplina = $disciplinasExistentes->get($key);
            
            if (!$disciplina) {
                try {
                    $disciplina = Disciplina::create([
                        'codigo' => $disc['codigo'],
                        'nome' => $disc['nome'],
                        'nivel_ensino' => 'PRIMARIO',
                        'carga_horaria_semanal' => $disc['carga_horaria_semanal'],
                        'ativa' => true,
                    ]);
                } catch (\Exception $e) {
                    // Se já existe por código, buscar e atualizar
                    $disciplina = Disciplina::where('codigo', $disc['codigo'])
                        ->where('nivel_ensino', 'PRIMARIO')
                        ->first();
                    
                    if (!$disciplina) {
                        // Se ainda não existe, criar com código diferente ou pular
                        continue;
                    }
                    
                    $disciplina->nome = $disc['nome'];
                    $disciplina->carga_horaria_semanal = $disc['carga_horaria_semanal'];
                    $disciplina->ativa = true;
                    $disciplina->save();
                }
            } else {
                // Atualizar se necessário
                $disciplina->nome = $disc['nome'];
                $disciplina->carga_horaria_semanal = $disc['carga_horaria_semanal'];
                $disciplina->ativa = true;
                $disciplina->save();
            }
            
            $disciplinas->push($disciplina);
        }
        
        foreach ($disciplinasSecundario as $disc) {
            $key = $disc['codigo'] . '_SECUNDARIO';
            $disciplina = $disciplinasExistentes->get($key);
            
            if (!$disciplina) {
                try {
                    $disciplina = Disciplina::create([
                        'codigo' => $disc['codigo'],
                        'nome' => $disc['nome'],
                        'nivel_ensino' => 'SECUNDARIO',
                        'carga_horaria_semanal' => $disc['carga_horaria_semanal'],
                        'ativa' => true,
                    ]);
                } catch (\Exception $e) {
                    // Se já existe por código, buscar e atualizar
                    $disciplina = Disciplina::where('codigo', $disc['codigo'])
                        ->where('nivel_ensino', 'SECUNDARIO')
                        ->first();
                    
                    if (!$disciplina) {
                        // Se ainda não existe, criar com código diferente ou pular
                        continue;
                    }
                    
                    $disciplina->nome = $disc['nome'];
                    $disciplina->carga_horaria_semanal = $disc['carga_horaria_semanal'];
                    $disciplina->ativa = true;
                    $disciplina->save();
                }
            } else {
                // Atualizar se necessário
                $disciplina->nome = $disc['nome'];
                $disciplina->carga_horaria_semanal = $disc['carga_horaria_semanal'];
                $disciplina->ativa = true;
                $disciplina->save();
            }
            
            $disciplinas->push($disciplina);
        }
        
        return $disciplinas;
    }
    
    private function criarTurmas()
    {
        $turmas = collect();
        
        // Turmas Primário (1ª a 6ª classe, A e B)
        for ($classe = 1; $classe <= 6; $classe++) {
            foreach (['A', 'B'] as $turno) {
                $turma = Turma::firstOrCreate(
                    [
                        'codigo' => "{$classe}A" === "1A" ? "1A" : "{$classe}{$turno}",
                        'nome' => "{$classe}ª Classe-{$turno}",
                        'ano_letivo_id' => $this->anoLetivo->id,
                    ],
                    [
                        'nivel_ensino' => 'PRIMARIO',
                        'capacidade_maxima' => 35,
                        'ativa' => true,
                    ]
                );
                $turmas->push($turma);
            }
        }
        
        // Turmas Secundário (7ª a 12ª classe, A e B)
        for ($classe = 7; $classe <= 12; $classe++) {
            foreach (['A', 'B'] as $turno) {
                $turma = Turma::firstOrCreate(
                    [
                        'codigo' => "{$classe}{$turno}",
                        'nome' => "{$classe}ª Classe-{$turno}",
                        'ano_letivo_id' => $this->anoLetivo->id,
                    ],
                    [
                        'nivel_ensino' => 'SECUNDARIO',
                        'capacidade_maxima' => 35,
                        'ativa' => true,
                    ]
                );
                $turmas->push($turma);
            }
        }
        
        return $turmas;
    }
    
    private function criarProfessores()
    {
        $professores = collect();
        $nomesProfessores = [
            'Carlos Mendes', 'Maria Silva', 'João Santos', 'Ana Costa', 'Pedro Oliveira',
            'Sofia Rodrigues', 'Miguel Fernandes', 'Beatriz Martins', 'Ricardo Pereira', 'Inês Alves',
            'António Machado', 'Patrícia Gomes', 'Paulo Sousa', 'Catarina Lima', 'Francisco Ferreira',
            'Teresa Ribeiro', 'José Carvalho', 'Helena Cunha', 'Manuel Dias', 'Rita Marques',
            'Luís Barbosa', 'Margarida Teixeira', 'Filipe Moreira', 'Joana Rocha', 'Nuno Araújo',
        ];
        
        $niveis = ['PRIMARIO', 'SECUNDARIO', 'AMBOS'];
        $timestamp = time();
        
        foreach ($nomesProfessores as $index => $nome) {
            $email = strtolower(str_replace(' ', '', $nome)) . $timestamp . $index . '@professor.sge.pt';
            
            $user = User::firstOrCreate(
                ['email' => $email],
                [
                    'name' => $nome,
                    'password' => Hash::make('password123'),
                    'tipo' => 'professor',
                    'must_change_password' => true,
                    'is_active' => true,
                ]
            );
            
            // Atribuir role de professor (via attach na relação roles)
            $roleProfessor = \App\Models\Role::where('name', 'professor')->first();
            if ($roleProfessor && !$user->roles->contains($roleProfessor->id)) {
                $user->roles()->attach($roleProfessor->id);
            }
            
            // Verificar se o professor já existe
            $professor = Professor::where('user_id', $user->id)->first();
            
            if (!$professor) {
                // Gerar número de funcionário único
                $numeroFuncionario = CodigoProfessorService::gerar();
                
                $professor = Professor::create([
                    'user_id' => $user->id,
                    'numero_funcionario' => $numeroFuncionario,
                    'nivel_ensino' => $niveis[$index % count($niveis)],
                ]);
            } else {
                // Atualizar nivel_ensino se necessário
                if ($professor->nivel_ensino !== $niveis[$index % count($niveis)]) {
                    $professor->nivel_ensino = $niveis[$index % count($niveis)];
                    $professor->save();
                }
            }
            
            $professores->push($professor);
        }
        
        return $professores;
    }
    
    private function alocarDisciplinasTurmas()
    {
        foreach ($this->turmas as $turma) {
            $disciplinasTurma = $this->disciplinas->where('nivel_ensino', $turma->nivel_ensino);
            
            foreach ($disciplinasTurma as $disciplina) {
                // Verificar se já existe relação
                $existe = DB::table('turma_disciplina')
                    ->where('turma_id', $turma->id)
                    ->where('disciplina_id', $disciplina->id)
                    ->where('ano_letivo_id', $this->anoLetivo->id)
                    ->exists();
                
                if (!$existe) {
                    DB::table('turma_disciplina')->insert([
                        'turma_id' => $turma->id,
                        'disciplina_id' => $disciplina->id,
                        'ano_letivo_id' => $this->anoLetivo->id,
                        'carga_horaria_semanal' => $disciplina->carga_horaria_semanal,
                        'created_at' => now(),
                        'updated_at' => now(),
                    ]);
                }
            }
        }
    }
    
    private function alocarProfessoresTurmasDisciplinas()
    {
        // Alocar professores às turmas-disciplinas
        // Um professor pode dar aula em múltiplas turmas
        foreach ($this->turmas as $turma) {
            $disciplinasTurma = $this->disciplinas->where('nivel_ensino', $turma->nivel_ensino);
            
            foreach ($disciplinasTurma as $disciplina) {
                // Buscar professores compatíveis (mesmo nível de ensino ou AMBOS)
                $professoresCompativel = $this->professores->filter(function($prof) use ($turma) {
                    return $prof->nivel_ensino === $turma->nivel_ensino || $prof->nivel_ensino === 'AMBOS';
                });
                
                if ($professoresCompativel->isEmpty()) {
                    continue;
                }
                
                // Selecionar um professor (pode ser o mesmo para várias turmas)
                $professor = $professoresCompativel->random();
                
                // Verificar se já existe relação
                $existe = DB::table('turma_disciplina_professor')
                    ->where('turma_id', $turma->id)
                    ->where('disciplina_id', $disciplina->id)
                    ->where('professor_id', $professor->id)
                    ->where('ano_letivo_id', $this->anoLetivo->id)
                    ->exists();
                
                if (!$existe) {
                    DB::table('turma_disciplina_professor')->insert([
                        'turma_id' => $turma->id,
                        'disciplina_id' => $disciplina->id,
                        'professor_id' => $professor->id,
                        'ano_letivo_id' => $this->anoLetivo->id,
                        'created_at' => now(),
                        'updated_at' => now(),
                    ]);
                }
            }
        }
    }
    
    private function criarAlunos($quantidade)
    {
        $alunos = collect();
        $anoIngresso = 2026;
        $timestamp = time();
        $index = 1;
        
        // Gerar nomes
        $nomesMasculinos = [
            'João', 'Pedro', 'Carlos', 'Miguel', 'Ricardo', 'António', 'Paulo', 'Francisco', 'José', 'Manuel',
            'Luís', 'Filipe', 'Nuno', 'André', 'Rui', 'Daniel', 'Gonçalo', 'Hugo', 'Rafael', 'Bruno',
        ];
        
        $nomesFemininos = [
            'Maria', 'Ana', 'Sofia', 'Beatriz', 'Inês', 'Patrícia', 'Catarina', 'Teresa', 'Helena', 'Margarida',
            'Joana', 'Rita', 'Rosa', 'Cristina', 'Sandra', 'Paula', 'Carla', 'Diana', 'Mónica', 'Lúcia',
        ];
        
        $sobrenomes = [
            'Silva', 'Santos', 'Costa', 'Oliveira', 'Rodrigues', 'Martins', 'Pereira', 'Alves', 'Fernandes', 'Gomes',
            'Sousa', 'Lima', 'Ferreira', 'Ribeiro', 'Carvalho', 'Cunha', 'Dias', 'Marques', 'Barbosa', 'Teixeira',
        ];
        
        $turmasPrimario = $this->turmas->where('nivel_ensino', 'PRIMARIO');
        $turmasSecundario = $this->turmas->where('nivel_ensino', 'SECUNDARIO');
        
        // Criar 200 alunos primário e 200 secundário
        for ($i = 0; $i < $quantidade; $i++) {
            $isPrimario = $i < ($quantidade / 2);
            $genero = rand(0, 1) === 0 ? 'M' : 'F';
            
            $nome = ($genero === 'M' ? $nomesMasculinos[array_rand($nomesMasculinos)] : $nomesFemininos[array_rand($nomesFemininos)])
                . ' ' . $sobrenomes[array_rand($sobrenomes)];
            
            $email = strtolower(str_replace(' ', '', $nome)) . $timestamp . $index . '@aluno.sge.pt';
            
            $user = User::create([
                'name' => $nome,
                'email' => $email,
                'password' => Hash::make('password123'),
                'tipo' => 'aluno',
                'must_change_password' => true,
                'is_active' => true,
            ]);
            
            $codigoEstudante = CodigoEstudanteService::gerar($anoIngresso);
            
            $aluno = Aluno::create([
                'user_id' => $user->id,
                'codigo_estudante' => $codigoEstudante,
                'genero' => $genero,
                'nome_encarregado' => 'Encarregado de ' . $nome,
                'telefone_encarregado' => '+258 8' . rand(1000000, 9999999),
                'modalidade_pagamento' => 'MENSAL',
            ]);
            
            // Alocar a uma turma do nível apropriado
            $turma = $isPrimario 
                ? $turmasPrimario->random() 
                : $turmasSecundario->random();
            
            $aluno->turmas()->attach($turma->id, ['ano_letivo_id' => $this->anoLetivo->id]);
            
            $alunos->push($aluno);
            $index++;
        }
        
        return $alunos;
    }
    
    private function criarPagamentosJaneiro($alunos)
    {
        $mes = 1;
        $ano = 2026;
        $criados = 0;
        
        foreach ($alunos as $aluno) {
            $valorMensal = $this->financeiroService->getValorMensalidadePorClasse($aluno, $this->anoLetivo->id);
            
            if ($valorMensal > 0) {
                $dataVencimento = \Carbon\Carbon::create($ano, $mes, 5);
                
                // Criar mensalidade de janeiro
                $financeiro = Financeiro::firstOrCreate(
                    [
                        'aluno_id' => $aluno->id,
                        'ano_letivo_id' => $this->anoLetivo->id,
                        'tipo' => 'Mensalidade',
                        'descricao' => "Mensalidade {$mes}/{$ano}",
                    ],
                    [
                        'valor' => $valorMensal,
                        'data_vencimento' => $dataVencimento,
                        'status' => 'PAGO',
                        'data_pagamento' => $dataVencimento->copy()->addDays(rand(1, 3)),
                        'valor_pago' => $valorMensal,
                    ]
                );
                
                // Criar pagamento
                $payment = Payment::create([
                    'aluno_id' => $aluno->id,
                    'financeiro_id' => $financeiro->id,
                    'reference' => 'PAG-' . time() . '-' . $aluno->id,
                    'metodo' => 'OUTRO',
                    'valor' => $valorMensal,
                    'status' => 'APROVADO',
                    'data_pagamento' => $financeiro->data_pagamento,
                    'observacoes' => 'Pagamento de janeiro - Gerado via seeder',
                ]);
                
                // Criar factura
                $dataEmissao = $financeiro->data_pagamento ?? $dataVencimento->copy()->addDays(rand(1, 3));
                
                $invoice = Invoice::create([
                    'aluno_id' => $aluno->id,
                    'payment_id' => $payment->id,
                    'ano_letivo_id' => $this->anoLetivo->id,
                    'numero_factura' => Invoice::gerarNumeroFactura(),
                    'codigo_verificacao' => Invoice::gerarCodigoVerificacao(),
                    'valor' => $valorMensal,
                    'status' => 'PAGA',
                    'data_emissao' => $dataEmissao,
                    'data_vencimento' => $dataVencimento,
                    'descricao' => "Mensalidade {$mes}/{$ano}",
                    'itens' => [
                        [
                            'descricao' => "Mensalidade {$mes}/{$ano}",
                            'valor' => $valorMensal,
                        ]
                    ],
                ]);
                
                $payment->invoice_id = $invoice->id;
                $payment->save();
                
                $criados++;
            }
        }
        
        $this->command->info("Criados {$criados} pagamentos de janeiro");
    }
    
    private function criarTrimestre($numero)
    {
        $dataInicio = match($numero) {
            1 => '2026-01-01',
            2 => '2026-04-01',
            3 => '2026-07-01',
            default => '2026-01-01',
        };
        
        $dataFim = match($numero) {
            1 => '2026-03-31',
            2 => '2026-06-30',
            3 => '2026-12-31',
            default => '2026-03-31',
        };
        
        $nome = match($numero) {
            1 => '1º Trimestre',
            2 => '2º Trimestre',
            3 => '3º Trimestre',
            default => '1º Trimestre',
        };
        
        return Trimestre::firstOrCreate(
            [
                'ano_letivo_id' => $this->anoLetivo->id,
                'numero' => $numero,
            ],
            [
                'nome' => $nome,
                'data_inicio' => $dataInicio,
                'data_fim' => $dataFim,
            ]
        );
    }
    
    private function criarHorarios()
    {
        // Criar horários básicos para cada turma
        // Desta forma simplificada, cada disciplina terá horários distribuídos
        $diasSemana = ['Segunda', 'Terça', 'Quarta', 'Quinta', 'Sexta'];
        $periodosManha = [
            ['inicio' => '08:00', 'fim' => '08:45'],
            ['inicio' => '08:45', 'fim' => '09:30'],
            ['inicio' => '10:00', 'fim' => '10:45'],
            ['inicio' => '10:45', 'fim' => '11:30'],
            ['inicio' => '11:30', 'fim' => '12:15'],
        ];
        
        $periodosTarde = [
            ['inicio' => '14:00', 'fim' => '14:45'],
            ['inicio' => '14:45', 'fim' => '15:30'],
            ['inicio' => '16:00', 'fim' => '16:45'],
            ['inicio' => '16:45', 'fim' => '17:30'],
        ];
        
        $horariosCriados = 0;
        
        foreach ($this->turmas as $turma) {
            // Buscar disciplinas da turma com professores alocados
            $turmaDisciplinas = DB::table('turma_disciplina_professor')
                ->where('turma_id', $turma->id)
                ->where('ano_letivo_id', $this->anoLetivo->id)
                ->get();
            
            foreach ($turmaDisciplinas as $turmaDisc) {
                $disciplina = $this->disciplinas->firstWhere('id', $turmaDisc->disciplina_id);
                if (!$disciplina) continue;
                
                // Calcular quantas aulas por semana (baseado na carga horária)
                // Assumindo aulas de 45 minutos
                $aulasPorSemana = (int) ceil($disciplina->carga_horaria_semanal / 45);
                
                // Distribuir aulas pelos dias da semana
                $periodosUsados = [];
                $diasUsados = [];
                
                for ($aula = 0; $aula < $aulasPorSemana && $aula < 10; $aula++) {
                    // Selecionar dia e período (evitar duplicatas no mesmo dia)
                    $dia = $diasSemana[$aula % count($diasSemana)];
                    $periodoIndex = (int) floor($aula / count($diasSemana));
                    
                    // Usar manhã para as primeiras aulas, tarde para as seguintes
                    $periodos = $periodoIndex < 5 ? $periodosManha : $periodosTarde;
                    $periodo = $periodos[$periodoIndex % count($periodos)] ?? $periodosManha[0];
                    
                    // Verificar se já existe horário para esta turma-disciplina-dia-hora
                    $existe = Horario::where('turma_id', $turma->id)
                        ->where('disciplina_id', $disciplina->id)
                        ->where('dia_semana', $dia)
                        ->where('hora_inicio', $periodo['inicio'])
                        ->where('ano_letivo_id', $this->anoLetivo->id)
                        ->exists();
                    
                    if (!$existe) {
                        Horario::create([
                            'turma_id' => $turma->id,
                            'disciplina_id' => $disciplina->id,
                            'professor_id' => $turmaDisc->professor_id,
                            'dia_semana' => $dia,
                            'hora_inicio' => $periodo['inicio'],
                            'hora_fim' => $periodo['fim'],
                            'sala' => 'Sala ' . rand(1, 20),
                            'ano_letivo_id' => $this->anoLetivo->id,
                        ]);
                        
                        $horariosCriados++;
                    }
                }
            }
        }
        
        $this->command->info("Criados {$horariosCriados} horários");
    }
    
    private function criarEventos()
    {
        $eventos = [
            ['titulo' => 'Início do Ano Letivo', 'data' => '2026-01-15', 'tipo' => 'Geral', 'descricao' => 'Cerimónia de abertura do ano letivo 2026'],
            ['titulo' => 'Dia da Mulher', 'data' => '2026-03-08', 'tipo' => 'Geral', 'descricao' => 'Celebração do Dia Internacional da Mulher'],
            ['titulo' => 'Dia do Trabalhador', 'data' => '2026-05-01', 'tipo' => 'Geral', 'descricao' => 'Feriado Nacional'],
            ['titulo' => 'Dia da Criança', 'data' => '2026-06-01', 'tipo' => 'Geral', 'descricao' => 'Celebração do Dia da Criança'],
            ['titulo' => 'Dia da Independência', 'data' => '2026-06-25', 'tipo' => 'Geral', 'descricao' => 'Feriado Nacional - Independência de Moçambique'],
            ['titulo' => 'Dia da Vitória', 'data' => '2026-09-07', 'tipo' => 'Geral', 'descricao' => 'Feriado Nacional'],
            ['titulo' => 'Dia da Revolução', 'data' => '2026-09-25', 'tipo' => 'Geral', 'descricao' => 'Feriado Nacional'],
            ['titulo' => 'Feira de Ciências', 'data' => '2026-04-20', 'tipo' => 'Geral', 'descricao' => 'Exposição de projetos científicos dos alunos'],
            ['titulo' => 'Festival de Artes', 'data' => '2026-05-15', 'tipo' => 'Geral', 'descricao' => 'Apresentações artísticas e culturais'],
            ['titulo' => 'Campeonato de Futebol', 'data' => '2026-06-10', 'tipo' => 'Geral', 'descricao' => 'Competição desportiva entre turmas'],
            ['titulo' => 'Dia do Professor', 'data' => '2026-10-05', 'tipo' => 'Geral', 'descricao' => 'Homenagem aos professores'],
            ['titulo' => 'Semana Cultural', 'data' => '2026-07-15', 'tipo' => 'Geral', 'descricao' => 'Semana dedicada à cultura moçambicana'],
            ['titulo' => 'Olimpíadas de Matemática', 'data' => '2026-03-20', 'tipo' => 'Disciplina', 'descricao' => 'Competição de matemática'],
            ['titulo' => 'Concurso de Leitura', 'data' => '2026-04-10', 'tipo' => 'Disciplina', 'descricao' => 'Concurso de leitura e interpretação'],
            ['titulo' => 'Visita de Estudo - Museu', 'data' => '2026-05-25', 'tipo' => 'Geral', 'descricao' => 'Visita educativa ao museu'],
            ['titulo' => 'Palestra sobre Saúde', 'data' => '2026-06-15', 'tipo' => 'Geral', 'descricao' => 'Palestra sobre saúde e higiene'],
            ['titulo' => 'Encerramento do 1º Trimestre', 'data' => '2026-03-31', 'tipo' => 'Geral', 'descricao' => 'Cerimónia de encerramento do primeiro trimestre'],
            ['titulo' => 'Encerramento do 2º Trimestre', 'data' => '2026-06-30', 'tipo' => 'Geral', 'descricao' => 'Cerimónia de encerramento do segundo trimestre'],
            ['titulo' => 'Cerimónia de Final de Ano', 'data' => '2026-12-15', 'tipo' => 'Geral', 'descricao' => 'Cerimónia de encerramento do ano letivo'],
            ['titulo' => 'Entrega de Diplomas', 'data' => '2026-12-20', 'tipo' => 'Geral', 'descricao' => 'Cerimónia de entrega de diplomas'],
        ];
        
        $eventosCriados = 0;
        
        foreach ($eventos as $evento) {
            $eventoModel = \App\Models\Evento::firstOrCreate(
                [
                    'titulo' => $evento['titulo'],
                    'data' => $evento['data'],
                    'ano_letivo_id' => $this->anoLetivo->id,
                ],
                [
                    'tipo' => $evento['tipo'],
                    'descricao' => $evento['descricao'],
                    'cancelado' => false,
                ]
            );
            
            if ($eventoModel->wasRecentlyCreated) {
                $eventosCriados++;
            }
        }
        
        $this->command->info("Criados {$eventosCriados} eventos");
    }
    
    private function criarNotasTrimestre($trimestre, $alunos)
    {
        $avaliacoesCriadas = 0;
        
        foreach ($this->turmas as $turma) {
            // Buscar alunos da turma
            $alunosTurma = $alunos->filter(function($aluno) use ($turma) {
                return $aluno->turmas()
                    ->wherePivot('turma_id', $turma->id)
                    ->wherePivot('ano_letivo_id', $this->anoLetivo->id)
                    ->exists();
            });
            
            // Buscar disciplinas da turma
            $disciplinasTurma = DB::table('turma_disciplina_professor')
                ->where('turma_id', $turma->id)
                ->where('ano_letivo_id', $this->anoLetivo->id)
                ->pluck('disciplina_id')
                ->unique();
            
            foreach ($alunosTurma as $aluno) {
                foreach ($disciplinasTurma as $disciplinaId) {
                    $disciplina = $this->disciplinas->firstWhere('id', $disciplinaId);
                    if (!$disciplina) continue;
                    
                    // Verificar se já existe avaliação
                    $existe = Avaliacao::where('aluno_id', $aluno->id)
                        ->where('turma_id', $turma->id)
                        ->where('disciplina_id', $disciplinaId)
                        ->where('trimestre_id', $trimestre->id)
                        ->where('ano_letivo_id', $this->anoLetivo->id)
                        ->exists();
                    
                    if (!$existe) {
                        $isSecundario = $turma->nivel_ensino === 'SECUNDARIO';
                        
                        if ($isSecundario) {
                            // Secundário: Teste1, Teste2, Avaliação Final
                            $teste1 = rand(8, 18) + (rand(0, 9) / 10);
                            $teste2 = rand(8, 18) + (rand(0, 9) / 10);
                            $avaliacaoFinal = rand(8, 18) + (rand(0, 9) / 10);
                            
                            // Média nacional = (Teste1 + Teste2 + 2*AF) / 4
                            $mediaNacional = round(($teste1 + $teste2 + 2 * $avaliacaoFinal) / 4, 1);
                            
                            Avaliacao::create([
                                'aluno_id' => $aluno->id,
                                'turma_id' => $turma->id,
                                'disciplina_id' => $disciplinaId,
                                'trimestre_id' => $trimestre->id,
                                'ano_letivo_id' => $this->anoLetivo->id,
                                'teste1' => $teste1,
                                'teste2' => $teste2,
                                'avaliacao_final' => $avaliacaoFinal,
                                'media_nacional' => $mediaNacional,
                            ]);
                        } else {
                            // Primário: A.Prática, ACS, AT
                            $aPratica = rand(8, 18) + (rand(0, 9) / 10);
                            $acs = rand(8, 18) + (rand(0, 9) / 10);
                            $at = rand(8, 18) + (rand(0, 9) / 10);
                            
                            // Média trimestral = (A.Prática + ACS + AT) / 3
                            $mt = round(($aPratica + $acs + $at) / 3, 1);
                            
                            Avaliacao::create([
                                'aluno_id' => $aluno->id,
                                'turma_id' => $turma->id,
                                'disciplina_id' => $disciplinaId,
                                'trimestre_id' => $trimestre->id,
                                'ano_letivo_id' => $this->anoLetivo->id,
                                'a_pratica' => $aPratica,
                                'acs' => $acs,
                                'at' => $at,
                                'mt' => $mt,
                            ]);
                        }
                        
                        $avaliacoesCriadas++;
                    }
                }
            }
        }
        
        $this->command->info("Criadas {$avaliacoesCriadas} avaliações do trimestre {$trimestre->numero}");
    }
}
