<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\Turma;
use App\Models\AnoLetivo;
use Illuminate\Support\Facades\DB;

class CriarTurmasEAlocarAlunosRemovidosSeeder extends Seeder
{
    public function run(): void
    {
        $this->command->info('=== CRIANDO NOVAS TURMAS E ALOCANDO ALUNOS REMOVIDOS ===');
        
        $anoLetivo = AnoLetivo::getAnoLetivoAtual();
        
        if (!$anoLetivo) {
            $this->command->error('Ano letivo atual não encontrado!');
            return;
        }
        
        $this->command->info("Usando ano letivo: {$anoLetivo->ano} (ID: {$anoLetivo->id})");
        
        // Buscar alunos que não estão em nenhuma turma do ano letivo atual
        $alunosSemTurma = DB::table('alunos')
            ->whereNotExists(function($query) use ($anoLetivo) {
                $query->select(DB::raw(1))
                    ->from('turma_aluno')
                    ->whereColumn('turma_aluno.aluno_id', 'alunos.id')
                    ->where('turma_aluno.ano_letivo_id', $anoLetivo->id);
            })
            ->get();
        
        $totalAlunosSemTurma = $alunosSemTurma->count();
        $this->command->info("Encontrados {$totalAlunosSemTurma} alunos sem turma");
        
        if ($totalAlunosSemTurma === 0) {
            $this->command->info('Nenhum aluno sem turma encontrado. Nada a fazer.');
            return;
        }
        
        // Agrupar alunos por classe (baseado nas turmas existentes)
        // Vamos criar turmas C, D, E, etc. para cada classe que tiver alunos sem turma
        $turmasExistentes = Turma::where('ano_letivo_id', $anoLetivo->id)
            ->where('ativa', true)
            ->get()
            ->groupBy(function($turma) {
                // Extrair número da classe do código ou nome
                if (preg_match('/^(\d+)/', $turma->codigo, $matches)) {
                    return (int)$matches[1];
                }
                if (preg_match('/(\d+)ª/', $turma->nome, $matches)) {
                    return (int)$matches[1];
                }
                return 0;
            });
        
        // Criar novas turmas e alocar alunos
        $turmasCriadas = 0;
        $alunosAlocados = 0;
        $capacidadeMaxima = 35;
        
        // Para cada classe (1 a 12)
        for ($classe = 1; $classe <= 12; $classe++) {
            // Determinar nível de ensino
            $nivelEnsino = $classe <= 6 ? 'PRIMARIO' : 'SECUNDARIO';
            
            // Contar quantos alunos sem turma temos para esta classe
            // Como não temos informação direta da classe do aluno, vamos distribuir proporcionalmente
            // ou criar turmas para todas as classes
            $letras = ['C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L'];
            $indiceLetra = 0;
            
            // Calcular quantos alunos precisam de turma para esta classe
            // Vamos assumir distribuição uniforme ou criar turmas baseado na necessidade
            $alunosPorClasse = (int) ceil($totalAlunosSemTurma / 12);
            $alunosParaEstaClasse = min($alunosPorClasse, $totalAlunosSemTurma - $alunosAlocados);
            
            if ($alunosParaEstaClasse > 0) {
                // Criar turmas até alocar todos os alunos desta classe
                $alunosRestantes = $alunosParaEstaClasse;
                
                while ($alunosRestantes > 0 && $indiceLetra < count($letras)) {
                    $letra = $letras[$indiceLetra];
                    $codigo = "{$classe}{$letra}";
                    $nome = "{$classe}ª Classe-{$letra}";
                    
                    // Verificar se a turma já existe
                    $turmaExistente = Turma::where('codigo', $codigo)
                        ->where('nome', $nome)
                        ->where('ano_letivo_id', $anoLetivo->id)
                        ->first();
                    
                    if (!$turmaExistente) {
                        // Criar nova turma
                        $turma = Turma::create([
                            'codigo' => $codigo,
                            'nome' => $nome,
                            'ano_letivo_id' => $anoLetivo->id,
                            'nivel_ensino' => $nivelEnsino,
                            'capacidade_maxima' => $capacidadeMaxima,
                            'ativa' => true,
                        ]);
                        
                        $this->command->info("Criada turma: {$codigo} - {$nome}");
                        $turmasCriadas++;
                    } else {
                        $turma = $turmaExistente;
                    }
                    
                    // Verificar quantos alunos já estão nesta turma
                    $alunosNaTurma = DB::table('turma_aluno')
                        ->where('turma_id', $turma->id)
                        ->where('ano_letivo_id', $anoLetivo->id)
                        ->count();
                    
                    $vagasDisponiveis = $capacidadeMaxima - $alunosNaTurma;
                    
                    if ($vagasDisponiveis > 0) {
                        // Buscar alunos sem turma para alocar
                        $alunosParaAlocar = DB::table('alunos')
                            ->whereNotExists(function($query) use ($anoLetivo) {
                                $query->select(DB::raw(1))
                                    ->from('turma_aluno')
                                    ->whereColumn('turma_aluno.aluno_id', 'alunos.id')
                                    ->where('turma_aluno.ano_letivo_id', $anoLetivo->id);
                            })
                            ->limit(min($vagasDisponiveis, $alunosRestantes))
                            ->pluck('id');
                        
                        // Alocar alunos à turma
                        foreach ($alunosParaAlocar as $alunoId) {
                            DB::table('turma_aluno')->insert([
                                'aluno_id' => $alunoId,
                                'turma_id' => $turma->id,
                                'ano_letivo_id' => $anoLetivo->id,
                                'created_at' => now(),
                                'updated_at' => now(),
                            ]);
                            
                            $alunosAlocados++;
                            $alunosRestantes--;
                        }
                        
                        $this->command->info("  → Alocados " . count($alunosParaAlocar) . " aluno(s) à turma {$codigo}");
                    }
                    
                    $indiceLetra++;
                    
                    // Se não há mais alunos para esta classe, passar para a próxima
                    if ($alunosRestantes <= 0) {
                        break;
                    }
                }
            }
        }
        
        // Se ainda houver alunos sem turma, criar turmas adicionais
        $alunosRestantes = DB::table('alunos')
            ->whereNotExists(function($query) use ($anoLetivo) {
                $query->select(DB::raw(1))
                    ->from('turma_aluno')
                    ->whereColumn('turma_aluno.aluno_id', 'alunos.id')
                    ->where('turma_aluno.ano_letivo_id', $anoLetivo->id);
            })
            ->count();
        
        if ($alunosRestantes > 0) {
            $this->command->info("Ainda restam {$alunosRestantes} alunos sem turma. Criando turmas adicionais...");
            
            // Distribuir os alunos restantes em turmas adicionais
            $classe = 1;
            $letras = ['C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P'];
            $indiceLetra = 0;
            
            while ($alunosRestantes > 0 && $classe <= 12) {
                $nivelEnsino = $classe <= 6 ? 'PRIMARIO' : 'SECUNDARIO';
                
                if ($indiceLetra >= count($letras)) {
                    $indiceLetra = 0;
                    $classe++;
                    if ($classe > 12) break;
                }
                
                $letra = $letras[$indiceLetra];
                $codigo = "{$classe}{$letra}";
                $nome = "{$classe}ª Classe-{$letra}";
                
                // Verificar se a turma já existe
                $turma = Turma::where('codigo', $codigo)
                    ->where('nome', $nome)
                    ->where('ano_letivo_id', $anoLetivo->id)
                    ->first();
                
                if (!$turma) {
                    $turma = Turma::create([
                        'codigo' => $codigo,
                        'nome' => $nome,
                        'ano_letivo_id' => $anoLetivo->id,
                        'nivel_ensino' => $nivelEnsino,
                        'capacidade_maxima' => $capacidadeMaxima,
                        'ativa' => true,
                    ]);
                    
                    $this->command->info("Criada turma adicional: {$codigo} - {$nome}");
                    $turmasCriadas++;
                }
                
                // Alocar alunos
                $alunosNaTurma = DB::table('turma_aluno')
                    ->where('turma_id', $turma->id)
                    ->where('ano_letivo_id', $anoLetivo->id)
                    ->count();
                
                $vagasDisponiveis = $capacidadeMaxima - $alunosNaTurma;
                
                if ($vagasDisponiveis > 0) {
                    $alunosParaAlocar = DB::table('alunos')
                        ->whereNotExists(function($query) use ($anoLetivo) {
                            $query->select(DB::raw(1))
                                ->from('turma_aluno')
                                ->whereColumn('turma_aluno.aluno_id', 'alunos.id')
                                ->where('turma_aluno.ano_letivo_id', $anoLetivo->id);
                        })
                        ->limit($vagasDisponiveis)
                        ->pluck('id');
                    
                    foreach ($alunosParaAlocar as $alunoId) {
                        DB::table('turma_aluno')->insert([
                            'aluno_id' => $alunoId,
                            'turma_id' => $turma->id,
                            'ano_letivo_id' => $anoLetivo->id,
                            'created_at' => now(),
                            'updated_at' => now(),
                        ]);
                        
                        $alunosAlocados++;
                        $alunosRestantes--;
                    }
                    
                    if (count($alunosParaAlocar) > 0) {
                        $this->command->info("  → Alocados " . count($alunosParaAlocar) . " aluno(s) à turma {$codigo}");
                    }
                }
                
                $indiceLetra++;
            }
        }
        
        $this->command->info('');
        $this->command->info("=== CONCLUÍDO ===");
        $this->command->info("Turmas criadas: {$turmasCriadas}");
        $this->command->info("Alunos alocados: {$alunosAlocados}");
        
        // Verificar se ainda há alunos sem turma
        $alunosSemTurmaFinal = DB::table('alunos')
            ->whereNotExists(function($query) use ($anoLetivo) {
                $query->select(DB::raw(1))
                    ->from('turma_aluno')
                    ->whereColumn('turma_aluno.aluno_id', 'alunos.id')
                    ->where('turma_aluno.ano_letivo_id', $anoLetivo->id);
            })
            ->count();
        
        if ($alunosSemTurmaFinal > 0) {
            $this->command->warn("Ainda restam {$alunosSemTurmaFinal} alunos sem turma. Execute o seeder novamente ou crie mais turmas manualmente.");
        } else {
            $this->command->info("Todos os alunos foram alocados com sucesso!");
        }
    }
}
