<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\User;
use App\Models\Aluno;
use App\Models\Professor;
use App\Models\Avaliacao;
use App\Models\Turma;
use App\Models\Disciplina;
use App\Models\Trimestre;
use App\Models\AnoLetivo;
use App\Services\CodigoEstudanteService;
use App\Services\AvaliacaoService;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\App;

class ResetAndPopulateSeeder extends Seeder
{
    public function run(): void
    {
        $this->command->info('Limpando dados existentes...');
        
        // Limpar avaliações
        DB::table('avaliacoes')->truncate();
        DB::table('turma_disciplina_professor')->truncate();
        DB::table('turma_aluno')->truncate();
        
        // Limpar alunos e seus usuários
        $alunos = Aluno::with('user')->get();
        foreach ($alunos as $aluno) {
            if ($aluno->user) {
                $aluno->user->delete();
            }
            $aluno->forceDelete();
        }
        
        // Limpar professores e seus usuários
        $professores = Professor::with('user')->get();
        foreach ($professores as $professor) {
            if ($professor->user) {
                $professor->user->delete();
            }
            $professor->forceDelete();
        }
        
        $this->command->info('Criando 50 alunos (30 mulheres, 20 homens)...');
        $this->criarAlunos();
        
        $this->command->info('Criando 10 professores (6 homens, 4 mulheres)...');
        $this->criarProfessores();
        
        $this->command->info('Criando avaliações com aprovações e reprovações...');
        $this->criarAvaliacoes();
        
        $this->command->info('Dados populados com sucesso!');
    }
    
    private function criarAlunos()
    {
        $nomesMulheres = [
            'Maria Santos', 'Ana Oliveira', 'Sofia Pereira', 'Beatriz Rodrigues', 'Inês Martins',
            'Mariana Ferreira', 'Carolina Lopes', 'Leonor Carvalho', 'Matilde Correia', 'Lara Moreira',
            'Joana Silva', 'Rita Costa', 'Catarina Mendes', 'Diana Fernandes', 'Eva Alves',
            'Patrícia Sousa', 'Teresa Gomes', 'Cristina Ribeiro', 'Sandra Pinto', 'Mónica Nunes',
            'Isabel Ferreira', 'Carla Oliveira', 'Diana Santos', 'Patrícia Costa', 'Teresa Mendes',
            'Sandra Pereira', 'Mónica Fernandes', 'Rita Alves', 'Cristina Sousa', 'Carla Gomes'
        ];
        
        $nomesHomens = [
            'João Silva', 'Pedro Costa', 'Carlos Mendes', 'Miguel Fernandes', 'Tiago Alves',
            'Rafael Sousa', 'Gonçalo Gomes', 'Diogo Ribeiro', 'Francisco Pinto', 'Tomás Nunes',
            'António Machado', 'Manuel Santos', 'José Silva', 'Ricardo Mendes', 'Paulo Fernandes',
            'Luís Alves', 'Nuno Sousa', 'Hugo Gomes', 'Bruno Ribeiro', 'Filipe Pinto'
        ];
        
        $anoAtual = date('Y');
        $anoIngresso = $anoAtual;
        $index = 1;
        $timestamp = time();
        
        // Criar 30 mulheres
        foreach ($nomesMulheres as $nome) {
            $email = strtolower(str_replace(' ', '', $nome)) . $timestamp . $index . '@aluno.sge.pt';
            
            $user = User::create([
                'name' => $nome,
                'email' => $email,
                'password' => Hash::make($anoAtual),
                'tipo' => 'aluno',
                'must_change_password' => true,
                'is_active' => true,
            ]);
            
            $codigoEstudante = CodigoEstudanteService::gerar($anoIngresso);
            
            Aluno::create([
                'user_id' => $user->id,
                'codigo_estudante' => $codigoEstudante,
                'telefone' => '+258 8' . rand(1000000, 9999999),
                'endereco' => 'Endereço ' . $index . ', Maputo',
                'data_nascimento' => now()->subYears(rand(10, 18))->subDays(rand(0, 365)),
                'genero' => 'F',
                'nome_encarregado' => 'Encarregado de ' . $nome,
                'telefone_encarregado' => '+258 8' . rand(1000000, 9999999),
            ]);
            
            $index++;
        }
        
        // Criar 20 homens
        foreach ($nomesHomens as $nome) {
            $email = strtolower(str_replace(' ', '', $nome)) . $timestamp . $index . '@aluno.sge.pt';
            
            $user = User::create([
                'name' => $nome,
                'email' => $email,
                'password' => Hash::make($anoAtual),
                'tipo' => 'aluno',
                'must_change_password' => true,
                'is_active' => true,
            ]);
            
            $codigoEstudante = CodigoEstudanteService::gerar($anoIngresso);
            
            Aluno::create([
                'user_id' => $user->id,
                'codigo_estudante' => $codigoEstudante,
                'telefone' => '+258 8' . rand(1000000, 9999999),
                'endereco' => 'Endereço ' . $index . ', Maputo',
                'data_nascimento' => now()->subYears(rand(10, 18))->subDays(rand(0, 365)),
                'genero' => 'M',
                'nome_encarregado' => 'Encarregado de ' . $nome,
                'telefone_encarregado' => '+258 8' . rand(1000000, 9999999),
            ]);
            
            $index++;
        }
    }
    
    private function criarProfessores()
    {
        $nomesHomens = [
            'António Machado', 'Manuel Santos', 'José Silva', 'Ricardo Mendes', 'Paulo Fernandes', 'Luís Alves'
        ];
        
        $nomesMulheres = [
            'Isabel Ferreira', 'Cristina Oliveira', 'Patrícia Costa', 'Teresa Rodrigues'
        ];
        
        $anoAtual = date('Y');
        $index = 1;
        $timestamp = time();
        
        // Criar 6 homens
        foreach ($nomesHomens as $nome) {
            $email = strtolower(str_replace(' ', '', $nome)) . $timestamp . $index . '@professor.sge.pt';
            
            $user = User::create([
                'name' => $nome,
                'email' => $email,
                'password' => Hash::make($anoAtual),
                'tipo' => 'professor',
                'must_change_password' => true,
                'is_active' => true,
            ]);
            
            Professor::create([
                'user_id' => $user->id,
                'numero_funcionario' => 'PROF' . str_pad($index, 4, '0', STR_PAD_LEFT),
                'telefone' => '+258 8' . rand(1000000, 9999999),
                'endereco' => 'Endereço ' . $index . ', Maputo',
                'data_nascimento' => now()->subYears(rand(25, 55))->subDays(rand(0, 365)),
                'genero' => 'M',
            ]);
            
            $index++;
        }
        
        // Criar 4 mulheres
        foreach ($nomesMulheres as $nome) {
            $email = strtolower(str_replace(' ', '', $nome)) . $timestamp . $index . '@professor.sge.pt';
            
            $user = User::create([
                'name' => $nome,
                'email' => $email,
                'password' => Hash::make($anoAtual),
                'tipo' => 'professor',
                'must_change_password' => true,
                'is_active' => true,
            ]);
            
            Professor::create([
                'user_id' => $user->id,
                'numero_funcionario' => 'PROF' . str_pad($index, 4, '0', STR_PAD_LEFT),
                'telefone' => '+258 8' . rand(1000000, 9999999),
                'endereco' => 'Endereço ' . $index . ', Maputo',
                'data_nascimento' => now()->subYears(rand(25, 55))->subDays(rand(0, 365)),
                'genero' => 'F',
            ]);
            
            $index++;
        }
    }
    
    private function criarAvaliacoes()
    {
        $anoLetivo = AnoLetivo::where('ativo', true)->first();
        
        if (!$anoLetivo) {
            $this->command->warn('Nenhum ano letivo ativo encontrado.');
            return;
        }
        
        $trimestres = Trimestre::where('ano_letivo_id', $anoLetivo->id)
            ->orderBy('numero')
            ->get();
        
        if ($trimestres->isEmpty()) {
            $this->command->warn('Nenhum trimestre encontrado.');
            return;
        }
        
        $turmas = Turma::where('ano_letivo_id', $anoLetivo->id)
            ->where('ativa', true)
            ->get();
        
        if ($turmas->isEmpty()) {
            $this->command->warn('Nenhuma turma encontrada.');
            return;
        }
        
        // Buscar disciplinas atribuídas às turmas
        $disciplinas = Disciplina::where('ativa', true)->get();
        
        if ($disciplinas->isEmpty()) {
            $this->command->warn('Nenhuma disciplina encontrada.');
            return;
        }
        
        $alunos = Aluno::all();
        
        foreach ($turmas as $turma) {
            // Alocar alguns alunos à turma
            $alunosTurma = $alunos->random(min(30, $alunos->count()));
            foreach ($alunosTurma as $aluno) {
                $turma->alunos()->syncWithoutDetaching([
                    $aluno->id => ['ano_letivo_id' => $anoLetivo->id]
                ]);
            }
            
            // Alocar professores às disciplinas
            $professores = Professor::all();
            $disciplinasTurma = DB::table('turma_disciplina')
                ->where('turma_id', $turma->id)
                ->where('ano_letivo_id', $anoLetivo->id)
                ->pluck('disciplina_id')
                ->toArray();
            
            if (empty($disciplinasTurma)) {
                // Se não houver disciplinas atribuídas, usar todas
                $disciplinasTurma = $disciplinas->pluck('id')->toArray();
            }
            
            foreach ($disciplinasTurma as $disciplinaId) {
                $professor = $professores->random();
                
                DB::table('turma_disciplina_professor')->insert([
                    'turma_id' => $turma->id,
                    'disciplina_id' => $disciplinaId,
                    'professor_id' => $professor->id,
                    'ano_letivo_id' => $anoLetivo->id,
                    'created_at' => now(),
                    'updated_at' => now(),
                ]);
            }
            
            // Criar avaliações para cada aluno, disciplina e trimestre
            $alunoIndex = 0;
            $totalAlunos = $alunosTurma->count();
            $limiteReprovados = (int)($totalAlunos * 0.35); // Aproximadamente 35% reprovados, 65% aprovados
            foreach ($alunosTurma as $aluno) {
                // Determinar se este aluno será reprovado em todas as disciplinas
                $isReprovado = $alunoIndex < $limiteReprovados;
                
                foreach ($disciplinasTurma as $disciplinaId) {
                    foreach ($trimestres as $trimestre) {
                        if ($isReprovado) {
                            // Notas abaixo de 10 para reprovação (0-9.9)
                            $a_pratica_1 = rand(0, 99) / 10;
                            $a_pratica_2 = rand(0, 99) / 10;
                            $acs_1 = rand(0, 99) / 10;
                            $acs_2 = rand(0, 99) / 10;
                            $at = rand(0, 99) / 10;
                        } else {
                            // Notas entre 10-20 para aprovação
                            $a_pratica_1 = rand(100, 200) / 10;
                            $a_pratica_2 = rand(100, 200) / 10;
                            $acs_1 = rand(100, 200) / 10;
                            $acs_2 = rand(100, 200) / 10;
                            $at = rand(100, 200) / 10;
                        }
                        
                        $dados = [
                            'aluno_id' => $aluno->id,
                            'disciplina_id' => $disciplinaId,
                            'turma_id' => $turma->id,
                            'trimestre_id' => $trimestre->id,
                            'ano_letivo_id' => $anoLetivo->id,
                            'modelo' => 'NACIONAL',
                            'a_pratica_1' => $a_pratica_1,
                            'a_pratica_2' => $a_pratica_2,
                            'acs_1' => $acs_1,
                            'acs_2' => $acs_2,
                            'at' => $at,
                        ];
                        
                        $avaliacaoService = App::make(AvaliacaoService::class);
                        $avaliacaoService->salvarAvaliacao($dados);
                    }
                }
                $alunoIndex++;
            }
        }
        
        $this->command->info('Avaliações criadas com sucesso!');
    }
}

