<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use App\Models\Aluno;
use App\Models\Professor;
use App\Models\Funcionario;
use App\Models\User;
use App\Models\Turma;
use App\Models\Disciplina;
use App\Models\Financeiro;
use App\Models\Payment;
use App\Models\Invoice;
use App\Models\Avaliacao;
use App\Models\Horario;
use App\Models\Evento;
use App\Models\Trabalho;
use App\Models\PlanoTrimestral;
use App\Models\Trimestre;
use App\Models\AnoLetivo;
use App\Models\ProfessorPayment;
use App\Models\Message;
use App\Models\Notification;
use Illuminate\Support\Facades\Storage;

class ResetarTodasTabelasSeeder extends Seeder
{
    /**
     * Resetar todas as tabelas do banco de dados
     * ATENÇÃO: Este seeder remove TODOS os dados, EXCETO admins e superadmins!
     */
    public function run(): void
    {
        $this->command->warn('⚠️  ATENÇÃO: Este seeder vai apagar TODOS os dados do banco de dados!');
        $this->command->warn('   EXCETO: Admins e Superadmins serão preservados.');
        $this->command->info('=== INICIANDO RESET COMPLETO DAS TABELAS ===');
        
        // Desabilitar verificação de foreign keys temporariamente
        DB::statement('SET FOREIGN_KEY_CHECKS=0;');
        
        try {
            // 1. Apagar tabelas pivot e relacionadas primeiro (preservando user_role para admins/superadmins)
            $this->command->info('1. Apagando tabelas pivot e relacionadas...');
            $this->limparTabela('trabalho_aluno');
            $this->limparTabela('turma_aluno');
            $this->limparTabela('turma_disciplina');
            $this->limparTabela('turma_disciplina_professor');
            
            // Preservar user_role apenas para admins/superadmins
            $this->command->info('   Preservando roles de admins/superadmins...');
            $adminUserIds = User::withTrashed()
                ->whereIn('tipo', ['admin', 'superadmin'])
                ->pluck('id');
            
            if ($adminUserIds->count() > 0) {
                // Deletar apenas user_role que não são de admins/superadmins
                DB::table('user_role')
                    ->whereNotIn('user_id', $adminUserIds)
                    ->delete();
                $this->command->info("   ✓ Roles de {$adminUserIds->count()} admin(s)/superadmin(s) preservadas");
            } else {
                $this->limparTabela('user_role');
            }
            
            $this->limparTabela('user_permission');
            // Não limpar role_permission para preservar as permissões do sistema
            
            // 2. Apagar registros financeiros
            $this->command->info('2. Apagando registros financeiros...');
            $this->limparTabela('financeiro');
            $this->limparTabela('payments');
            $this->limparTabela('invoices');
            $this->limparTabela('professor_payments');
            
            // 3. Apagar avaliações e trabalhos
            $this->command->info('3. Apagando avaliações e trabalhos...');
            $this->limparTabela('avaliacoes');
            $this->limparTabela('trabalhos');
            $this->limparTabela('planos_trimestrais');
            $this->limparTabela('cambridge_grade_boundaries');
            
            // 4. Apagar horários e eventos
            $this->command->info('4. Apagando horários e eventos...');
            $this->limparTabela('horarios');
            $this->limparTabela('eventos');
            
            // 5. Apagar mensagens e notificações
            $this->command->info('5. Apagando mensagens e notificações...');
            $this->limparTabela('messages');
            $this->limparTabela('notifications');
            
            // 6. Apagar alunos (com hard delete)
            $this->command->info('6. Apagando alunos...');
            $alunos = Aluno::withTrashed()->get();
            foreach ($alunos as $aluno) {
                // Remover foto se existir
                if ($aluno->foto && Storage::disk('public')->exists($aluno->foto)) {
                    Storage::disk('public')->delete($aluno->foto);
                }
                $aluno->forceDelete();
            }
            
            // 7. Apagar professores (com hard delete)
            $this->command->info('7. Apagando professores...');
            $professores = Professor::withTrashed()->get();
            foreach ($professores as $professor) {
                // Remover documento de identificação se existir
                if ($professor->documento_identificacao && Storage::disk('public')->exists($professor->documento_identificacao)) {
                    Storage::disk('public')->delete($professor->documento_identificacao);
                }
                $professor->forceDelete();
            }
            
            // 8. Apagar funcionários (com hard delete, exceto admins/superadmins)
            $this->command->info('8. Apagando funcionários (preservando admins/superadmins)...');
            $funcionarios = Funcionario::withTrashed()->get();
            $funcionariosPreservados = 0;
            foreach ($funcionarios as $funcionario) {
                // Verificar se o funcionário é admin ou superadmin
                if ($funcionario->user && in_array($funcionario->user->tipo, ['admin', 'superadmin'])) {
                    $funcionariosPreservados++;
                    continue; // Não deletar
                }
                $funcionario->forceDelete();
            }
            if ($funcionariosPreservados > 0) {
                $this->command->info("   ✓ {$funcionariosPreservados} funcionário(s) admin/superadmin preservado(s)");
            }
            
            // 9. Apagar turmas (com hard delete)
            $this->command->info('9. Apagando turmas...');
            $turmas = Turma::withTrashed()->get();
            foreach ($turmas as $turma) {
                $turma->forceDelete();
            }
            
            // 10. Apagar disciplinas (com hard delete)
            $this->command->info('10. Apagando disciplinas...');
            $disciplinas = Disciplina::withTrashed()->get();
            foreach ($disciplinas as $disciplina) {
                $disciplina->forceDelete();
            }
            
            // 11. Apagar trimestres
            $this->command->info('11. Apagando trimestres...');
            $this->limparTabela('trimestres');
            
            // 12. Apagar anos letivos (manter apenas se necessário)
            $this->command->info('12. Apagando anos letivos...');
            $this->limparTabela('ano_letivo');
            
            // 13. Apagar usuários (com hard delete, exceto admins/superadmins)
            $this->command->info('13. Apagando usuários (preservando admins/superadmins)...');
            $users = User::withTrashed()->get();
            $usersPreservados = 0;
            foreach ($users as $user) {
                // Verificar se o usuário é admin ou superadmin
                if (in_array($user->tipo, ['admin', 'superadmin'])) {
                    $usersPreservados++;
                    continue; // Não deletar
                }
                $user->forceDelete();
            }
            if ($usersPreservados > 0) {
                $this->command->info("   ✓ {$usersPreservados} usuário(s) admin/superadmin preservado(s)");
            }
            
            // 14. Apagar outras tabelas
            $this->command->info('14. Apagando outras tabelas...');
            $this->limparTabela('auditoria');
            $this->limparTabela('login_attempts');
            $this->limparTabela('sessions');
            
            // 15. Limpar tabelas de configuração (opcional - descomente se quiser limpar)
            // $this->command->info('15. Limpando configurações...');
            // $this->limparTabela('configuracoes');
            // $this->limparTabela('temas');
            
            // 16. Limpar tabelas de roles e permissions (opcional - descomente se quiser limpar)
            // $this->command->info('16. Limpando roles e permissions...');
            // $this->limparTabela('roles');
            // $this->limparTabela('permissions');
            
            // Reabilitar verificação de foreign keys
            DB::statement('SET FOREIGN_KEY_CHECKS=1;');
            
            $this->command->info('');
            $this->command->info('✅ === RESET COMPLETO CONCLUÍDO ===');
            $this->command->info('Todas as tabelas foram limpas com sucesso!');
            $this->command->info('');
            $this->command->info('💡 Dica: Execute os seeders para popular novamente:');
            $this->command->info('   php artisan db:seed --class=RolePermissionSeeder');
            $this->command->info('   php artisan db:seed --class=SuperadminSeeder');
            $this->command->info('   php artisan db:seed --class=AdminSeeder');
            $this->command->info('   php artisan db:seed --class=AnoLetivoSeeder');
            $this->command->info('   php artisan db:seed --class=DisciplinaSeeder');
            
        } catch (\Exception $e) {
            DB::statement('SET FOREIGN_KEY_CHECKS=1;');
            $this->command->error('❌ Erro ao resetar tabelas: ' . $e->getMessage());
            $this->command->error('Stack trace: ' . $e->getTraceAsString());
            throw $e;
        }
    }
    
    /**
     * Limpar uma tabela específica
     */
    private function limparTabela(string $tableName): void
    {
        if (Schema::hasTable($tableName)) {
            $count = DB::table($tableName)->count();
            if ($count > 0) {
                DB::table($tableName)->truncate();
                $this->command->info("   ✓ {$tableName}: {$count} registros removidos");
            } else {
                $this->command->info("   - {$tableName}: já estava vazia");
            }
        } else {
            $this->command->warn("   ⚠ {$tableName}: tabela não existe");
        }
    }
}
