# Script para corrigir problemas de instalação no Windows/XAMPP

Write-Host "=== Verificando e Corrigindo Problemas de Instalação ===" -ForegroundColor Cyan

# 1. Verificar extensão ZIP
Write-Host "`n1. Verificando extensão ZIP do PHP..." -ForegroundColor Yellow
$phpIniPath = "C:\xampp\php\php.ini"

if (Test-Path $phpIniPath) {
    $phpIniContent = Get-Content $phpIniPath -Raw
    
    if ($phpIniContent -match ";extension=zip") {
        Write-Host "   [X] Extensão ZIP está desativada" -ForegroundColor Red
        Write-Host "   [!] Por favor, abra C:\xampp\php\php.ini e remova o ';' de ';extension=zip'" -ForegroundColor Yellow
        Write-Host "   [!] Depois, reinicie o Apache no XAMPP" -ForegroundColor Yellow
    } elseif ($phpIniContent -match "extension=zip") {
        Write-Host "   [OK] Extensão ZIP está ativada" -ForegroundColor Green
    } else {
        Write-Host "   [!] Extensão ZIP não encontrada no php.ini" -ForegroundColor Yellow
    }
} else {
    Write-Host "   [X] Arquivo php.ini não encontrado em: $phpIniPath" -ForegroundColor Red
}

# 2. Verificar Git
Write-Host "`n2. Verificando Git..." -ForegroundColor Yellow
try {
    $gitVersion = git --version 2>&1
    if ($LASTEXITCODE -eq 0) {
        Write-Host "   [OK] Git instalado: $gitVersion" -ForegroundColor Green
    } else {
        Write-Host "   [X] Git não encontrado" -ForegroundColor Red
        Write-Host "   [!] Baixe e instale: https://git-scm.com/download/win" -ForegroundColor Yellow
    }
} catch {
    Write-Host "   [X] Git não encontrado no PATH" -ForegroundColor Red
    Write-Host "   [!] Baixe e instale: https://git-scm.com/download/win" -ForegroundColor Yellow
    Write-Host "   [!] Durante a instalação, marque 'Add Git to PATH'" -ForegroundColor Yellow
}

# 3. Verificar Node.js
Write-Host "`n3. Verificando Node.js..." -ForegroundColor Yellow
try {
    $nodeVersion = node --version 2>&1
    if ($LASTEXITCODE -eq 0) {
        Write-Host "   [OK] Node.js instalado: $nodeVersion" -ForegroundColor Green
    } else {
        Write-Host "   [X] Node.js não encontrado" -ForegroundColor Red
        Write-Host "   [!] Baixe e instale: https://nodejs.org/" -ForegroundColor Yellow
    }
} catch {
    Write-Host "   [X] Node.js não encontrado no PATH" -ForegroundColor Red
    Write-Host "   [!] Baixe e instale: https://nodejs.org/" -ForegroundColor Yellow
}

# 4. Verificar extensões PHP
Write-Host "`n4. Verificando extensões PHP..." -ForegroundColor Yellow
$requiredExtensions = @("zip", "mbstring", "openssl", "pdo_mysql", "tokenizer", "xml", "ctype", "json", "fileinfo")
$missingExtensions = @()

foreach ($ext in $requiredExtensions) {
    $result = php -m 2>&1 | Select-String -Pattern "^$ext$"
    if ($result) {
        Write-Host "   [OK] $ext" -ForegroundColor Green
    } else {
        Write-Host "   [X] $ext - FALTANDO" -ForegroundColor Red
        $missingExtensions += $ext
    }
}

if ($missingExtensions.Count -gt 0) {
    Write-Host "`n   [!] Extensões faltando: $($missingExtensions -join ', ')" -ForegroundColor Yellow
    Write-Host "   [!] Ative-as no arquivo C:\xampp\php\php.ini" -ForegroundColor Yellow
    Write-Host "   [!] Procure por ';extension=$($missingExtensions[0])' e remova o ';'" -ForegroundColor Yellow
}

# 5. Resumo
Write-Host "`n=== RESUMO ===" -ForegroundColor Cyan
Write-Host "Após corrigir os problemas acima:" -ForegroundColor Yellow
Write-Host "1. Reinicie o terminal/PowerShell" -ForegroundColor White
Write-Host "2. Execute: composer install" -ForegroundColor White
Write-Host "3. Execute: npm install" -ForegroundColor White
Write-Host "4. Configure o arquivo .env" -ForegroundColor White
Write-Host "5. Execute: php artisan migrate --seed" -ForegroundColor White

Write-Host "`nPressione qualquer tecla para sair..."
$null = $Host.UI.RawUI.ReadKey("NoEcho,IncludeKeyDown")

