<?php

use Illuminate\Support\Facades\Route;
use App\Http\Controllers\AuthController;
use App\Http\Controllers\DashboardController;
use App\Http\Controllers\SuperadminController;
use App\Http\Controllers\AdminController;
use App\Http\Controllers\AvaliacaoController;
use App\Http\Controllers\FinanceiroController;
use App\Http\Controllers\TrabalhoController;
use App\Http\Controllers\HorarioController;
use App\Http\Controllers\EventoController;
use App\Http\Controllers\ProfessorController;
use App\Http\Controllers\Professor\TPCPlanoController;
use App\Http\Controllers\AlunoController;
use App\Http\Controllers\FuncionarioController;
use App\Http\Controllers\PasswordResetController;
use App\Http\Controllers\LanguageController;
use App\Http\Controllers\ChatController;
use App\Http\Controllers\NotificationController;
use App\Http\Controllers\AtividadeController;

// Rotas públicas
Route::get('/language/{locale}', [LanguageController::class, 'switch'])->name('language.switch');
Route::get('/', function () {
    return redirect()->route('login');
});

// Rotas públicas de inscrição
Route::get('/inscrever-se', [App\Http\Controllers\PublicInscricaoController::class, 'showForm'])->name('public.inscricao');
Route::post('/inscrever-se', [App\Http\Controllers\PublicInscricaoController::class, 'store'])->name('public.inscricao.store');
Route::get('/inscricao-sucesso', [App\Http\Controllers\PublicInscricaoController::class, 'success'])->name('public.inscricao.success');

Route::get('/login', [AuthController::class, 'showLoginForm'])->name('login');
Route::post('/login', [AuthController::class, 'login']);
Route::post('/logout', [AuthController::class, 'logout'])->name('logout');

Route::get('/forgot-password', [AuthController::class, 'showForgotPasswordForm'])->name('password.request');
Route::post('/forgot-password', [AuthController::class, 'sendResetLink'])->name('password.email');

// Rotas de reset de senha (quando o usuário clica no link do email)
Route::get('/reset-password/{token}', function ($token) {
    return view('auth.reset-password', ['token' => $token]);
})->name('password.reset');

Route::post('/reset-password', [AuthController::class, 'resetPassword'])->name('password.update');

// Rotas autenticadas
Route::middleware(['auth', 'force.password.change'])->group(function () {
    // Rota de teste de permissões (remover depois)
    Route::get('/teste-permissoes', function() {
        $user = auth()->user();
        $permissoes = [
            'gerir_alunos', 'gerir_professores', 'gerir_turmas', 'gerir_disciplinas',
            'gerir_mensalidades', 'gerir_horarios', 'gerir_eventos', 'gerir_planos',
            'gerir_avaliacoes', 'gerir_inscricoes',
        ];
        
        $resultado = [];
        foreach ($permissoes as $perm) {
            $resultado[$perm] = $user->podeGerir($perm);
        }
        
        return view('teste-permissoes', [
            'user' => $user,
            'permissoes' => $resultado,
            'funcionario' => $user->funcionario ?? null,
        ]);
    })->name('teste.permissoes');
    
    Route::get('/dashboard', [DashboardController::class, 'index'])->name('dashboard');
    
    // Alteração de senha
    Route::get('/password/change', [AuthController::class, 'showPasswordChangeForm'])->name('password.change');
    Route::post('/password/change', [AuthController::class, 'changePassword'])->name('password.update');

    // Superadmin
    Route::prefix('superadmin')->middleware('role:superadmin')->name('superadmin.')->group(function () {
        // Gestão de Admins
        Route::get('/admins', [SuperadminController::class, 'indexAdmins'])->name('admins.index');
        Route::get('/admins/create', [SuperadminController::class, 'createAdmin'])->name('admins.create');
        Route::post('/admins', [SuperadminController::class, 'storeAdmin'])->name('admins.store');
        Route::get('/admins/{user}/edit', [SuperadminController::class, 'editAdmin'])->name('admins.edit');
        Route::put('/admins/{user}', [SuperadminController::class, 'updateAdmin'])->name('admins.update');
        Route::post('/admins/{user}/toggle-block', [SuperadminController::class, 'toggleBlockAdmin'])->name('admins.toggle-block');
        Route::delete('/admins/{user}', [SuperadminController::class, 'destroyAdmin'])->name('admins.destroy');

        // Configurações
        Route::get('/configuracoes', [SuperadminController::class, 'configuracoes'])->name('configuracoes');
        Route::post('/configuracoes', [SuperadminController::class, 'updateConfiguracoes'])->name('configuracoes.update');
        Route::get('/configuracoes-cabecalho', [SuperadminController::class, 'configuracoesCabecalho'])->name('configuracoes.cabecalho');
        Route::post('/configuracoes-cabecalho', [SuperadminController::class, 'updateConfiguracoesCabecalho'])->name('configuracoes.cabecalho.update');

        // Trimestres
        Route::get('/trimestres', [SuperadminController::class, 'trimestres'])->name('trimestres');
        Route::post('/trimestres', [SuperadminController::class, 'storeTrimestre'])->name('trimestres.store');

        // Logs e Auditoria
        Route::get('/logs', [SuperadminController::class, 'logs'])->name('logs.index');
        Route::get('/logs/erros', [SuperadminController::class, 'logsErros'])->name('logs.erros');
        Route::get('/logs/{id}', [SuperadminController::class, 'logsDetalhes'])->name('logs.detalhes');

        // Sistema de Backup
        Route::get('/backups', [SuperadminController::class, 'backups'])->name('backups.index');
        Route::get('/backups-list', fn () => redirect()->route('superadmin.backups.index'))->name('backups'); // alias
        Route::post('/backups/criar', [SuperadminController::class, 'criarBackup'])->name('backups.criar');
        Route::post('/backups/limpar', [SuperadminController::class, 'limparBackupsAntigos'])->name('backups.limpar');
        Route::get('/backups/{nome}/download', [SuperadminController::class, 'downloadBackup'])->name('backups.download');
        Route::delete('/backups/{nome}', [SuperadminController::class, 'deletarBackup'])->name('backups.deletar');
        
        // Google Drive
        Route::get('/backups/google-drive', [SuperadminController::class, 'configuracoesGoogleDrive'])->name('backups.google-drive');
        Route::post('/backups/google-drive', [SuperadminController::class, 'updateConfiguracoesGoogleDrive'])->name('backups.google-drive.update');
        Route::get('/backups/google-drive/auth', [SuperadminController::class, 'googleDriveAuth'])->name('backups.google-drive.auth');
        Route::get('/backups/google-drive/callback', [SuperadminController::class, 'googleDriveCallback'])->name('backups.google-drive.callback');
        
        // Modo de Manutenção
        Route::get('/manutencao', [SuperadminController::class, 'manutencao'])->name('manutencao');
        Route::post('/manutencao', [SuperadminController::class, 'updateManutencao'])->name('manutencao.update');

        // Atividade / Logins
        Route::get('/atividade-logins', [AtividadeController::class, 'index'])->name('atividade-logins');

        // Cambridge Grade Boundaries
    });

    // Admin (Superadmin também tem acesso, e Funcionários com permissões)
    Route::prefix('admin')->middleware('role:admin,superadmin,funcionario')->name('admin.')->group(function () {
        // Dashboard de Gestão
        Route::get('/gestao', function() {
            return view('admin.index');
        })->name('gestao');

        // Dashboard de Pagamentos
        Route::get('/pagamentos', [\App\Http\Controllers\Admin\PaymentsDashboardController::class, 'index'])->name('payments.index');

        // Atividade / Logins (último login, online, tentativas)
        Route::get('/atividade-logins', [AtividadeController::class, 'index'])->name('atividade-logins');

        // Configurações - Controle de Lançamento de Notas
        Route::get('/configuracoes/lancamento-notas', [AdminController::class, 'configuracoesLancamentoNotas'])->name('configuracoes.lancamento-notas');
        Route::post('/configuracoes/lancamento-notas', [AdminController::class, 'updateConfiguracoesLancamentoNotas'])->name('configuracoes.lancamento-notas.update');

        // Alunos
        Route::get('/alunos', [AdminController::class, 'indexAlunos'])->name('alunos.index');
        Route::get('/alunos/multiplas-turmas', [AdminController::class, 'alunosMultiplasTurmas'])->name('alunos.multiplas-turmas');
        Route::get('/alunos/create', [AdminController::class, 'createAluno'])->name('alunos.create');
        Route::post('/alunos', [AdminController::class, 'storeAluno'])->name('alunos.store');
        Route::get('/alunos/folhetos-turma', [AdminController::class, 'folhetosTurma'])->name('alunos.folhetos-turma');
        Route::get('/alunos/download-folheto', [AdminController::class, 'downloadFolhetoAluno'])->name('alunos.download-folheto');
        Route::get('/alunos/inscritos', [AdminController::class, 'alunosInscritos'])->name('alunos.inscritos');
        
        Route::get('/cartoes-estudantes', [AdminController::class, 'cartoesEstudantesIndex'])->name('cartoes-estudantes.index');
        Route::get('/cartoes-estudantes/imprimir', [AdminController::class, 'cartoesEstudantesImprimir'])->name('cartoes-estudantes.imprimir');
        Route::get('/cartoes-estudantes/download', [AdminController::class, 'cartoesEstudantesDownload'])->name('cartoes-estudantes.download');
        Route::get('/cartoes-estudantes/filtros', [AdminController::class, 'cartoesEstudantesFiltros'])->name('cartoes-estudantes.filtros');
        
        // Inscrições (Alunos Aprovados) - Rotas específicas ANTES da rota genérica {aluno}
        Route::get('/inscricoes', [AdminController::class, 'inscricoes'])->name('inscricoes.index');
        Route::get('/inscricoes/renovados', [AdminController::class, 'alunosRenovados'])->name('inscricoes.renovados');
        Route::get('/inscricoes/{aluno}/renovar', [AdminController::class, 'renovarAluno'])->name('inscricoes.renovar');
        Route::post('/inscricoes/{aluno}/renovar', [AdminController::class, 'processarRenovacaoAluno'])->name('inscricoes.renovar.processar');
        Route::get('/alunos/{aluno}/alocar', [AdminController::class, 'alocarAluno'])->name('alunos.alocar');
        Route::post('/alunos/{aluno}/alocar', [AdminController::class, 'storeAlocarAluno'])->name('alunos.alocar.store');
        Route::get('/alunos/{aluno}', [AdminController::class, 'showAluno'])->name('alunos.show');
        Route::get('/alunos/{aluno}/view', [AdminController::class, 'viewAluno'])->name('alunos.view');
        Route::get('/alunos/{aluno}/edit', [AdminController::class, 'editAluno'])->name('alunos.edit');
        Route::put('/alunos/{aluno}', [AdminController::class, 'updateAluno'])->name('alunos.update');
        Route::delete('/alunos/{aluno}', [AdminController::class, 'destroyAluno'])->name('alunos.destroy');
        Route::patch('/alunos/{aluno}/toggle-status', [AdminController::class, 'toggleStatusAluno'])->name('alunos.toggle-status');
        Route::get('/alunos/{aluno}/imprimir-matricula', [AdminController::class, 'imprimirMatricula'])->name('alunos.imprimir-matricula');
        Route::get('/alunos/{aluno}/imprimir-recibo', [AdminController::class, 'imprimirRecibo'])->name('alunos.imprimir-recibo');
        Route::get('/alunos/{aluno}/imprimir-cartao', [AdminController::class, 'imprimirCartao'])->name('alunos.imprimir-cartao');
        Route::get('/alunos/{aluno}/download-cartao', [AdminController::class, 'downloadCartao'])->name('alunos.download-cartao');
        Route::get('/alunos/{aluno}/ficha-inscricao', [AdminController::class, 'visualizarFichaInscricao'])->name('alunos.ficha-inscricao');
        Route::get('/alunos/{aluno}/ficha-inscricao/download', [AdminController::class, 'fichaInscricao'])->name('alunos.ficha-inscricao.download');
        Route::get('/alunos/{aluno}/folheto', [AdminController::class, 'folhetoAluno'])->name('alunos.folheto');
        
        // Inscrições pendentes
        Route::get('/inscricoes/pendentes', [AdminController::class, 'inscricoesPendentes'])->name('inscricoes.pendentes');
        Route::get('/inscricoes/{aluno}/visualizar', [AdminController::class, 'visualizarInscricao'])->name('inscricoes.visualizar');
        Route::post('/inscricoes/{aluno}/aprovar', [AdminController::class, 'aprovarInscricao'])->name('inscricoes.aprovar');
        Route::post('/inscricoes/{aluno}/rejeitar', [AdminController::class, 'rejeitarInscricao'])->name('inscricoes.rejeitar');

        // Professores
        Route::get('/professores', [AdminController::class, 'indexProfessores'])->name('professores.index');
        Route::get('/professores/create', [AdminController::class, 'createProfessor'])->name('professores.create');
        Route::post('/professores', [AdminController::class, 'storeProfessor'])->name('professores.store');
        Route::get('/professores/{professor}', [AdminController::class, 'showProfessor'])->name('professores.show');
        Route::get('/professores/{professor}/edit', [AdminController::class, 'editProfessor'])->name('professores.edit');
        Route::put('/professores/{professor}', [AdminController::class, 'updateProfessor'])->name('professores.update');
        Route::patch('/professores/{professor}/valor-hora', [AdminController::class, 'updateValorHora'])->name('professores.update-valor-hora');
        Route::delete('/professores/{professor}', [AdminController::class, 'destroyProfessor'])->name('professores.destroy');
        Route::patch('/professores/{professor}/toggle-status', [AdminController::class, 'toggleStatusProfessor'])->name('professores.toggle-status');
        Route::get('/professores/{professor}/imprimir-cracha', [AdminController::class, 'imprimirCracha'])->name('professores.imprimir-cracha');

        // Crachás de Professores (emitir + imprimir)
        Route::get('/crachas-professores', [AdminController::class, 'cartoesProfessoresIndex'])->name('cartoes-professores.index');
        Route::get('/crachas-professores/filtros', [AdminController::class, 'cartoesProfessoresFiltros'])->name('cartoes-professores.filtros');
        Route::get('/crachas-professores/imprimir', [AdminController::class, 'cartoesProfessoresImprimir'])->name('cartoes-professores.imprimir');

        // Presença por QR (scanner em telefone/tablet)
        Route::get('/presenca-professor', [AdminController::class, 'indexPresencaProfessor'])->name('presenca-professor.index');
        Route::get('/presenca-professor/scanner', [AdminController::class, 'scannerPresencaProfessor'])->name('presenca-professor.scanner');
        Route::post('/presenca-professor/marcar', [AdminController::class, 'marcarPresencaProfessor'])->name('presenca-professor.marcar');
        
        // Pagamentos de Professores
        Route::prefix('professor-payments')->name('professor-payments.')->group(function () {
            Route::get('/', [\App\Http\Controllers\Admin\ProfessorPaymentController::class, 'index'])->name('index');
            Route::get('/create', [\App\Http\Controllers\Admin\ProfessorPaymentController::class, 'create'])->name('create');
            Route::post('/calcular', [\App\Http\Controllers\Admin\ProfessorPaymentController::class, 'calcular'])->name('calcular');
            Route::post('/', [\App\Http\Controllers\Admin\ProfessorPaymentController::class, 'store'])->name('store');
            Route::put('/{payment}', [\App\Http\Controllers\Admin\ProfessorPaymentController::class, 'update'])->name('update');
            Route::post('/{payment}/marcar-pago', [\App\Http\Controllers\Admin\ProfessorPaymentController::class, 'marcarComoPago'])->name('marcar-pago');
            Route::post('/marcar-todos-pago', [\App\Http\Controllers\Admin\ProfessorPaymentController::class, 'marcarTodosComoPago'])->name('marcar-todos-pago');
            Route::post('/{payment}/cancelar', [\App\Http\Controllers\Admin\ProfessorPaymentController::class, 'cancelar'])->name('cancelar');
            Route::delete('/{payment}', [\App\Http\Controllers\Admin\ProfessorPaymentController::class, 'destroy'])->name('destroy');
            Route::get('/folha-salario', [\App\Http\Controllers\Admin\ProfessorPaymentController::class, 'gerarFolhaSalario'])->name('folha-salario');
            Route::get('/folha-salario/visualizar', [\App\Http\Controllers\Admin\ProfessorPaymentController::class, 'visualizarFolhaSalario'])->name('folha-salario.visualizar');
            Route::post('/gerar-mes-seguinte', [\App\Http\Controllers\Admin\ProfessorPaymentController::class, 'gerarMesSeguinte'])->name('gerar-mes-seguinte');
        });

        // Pagamentos de Funcionários
        Route::prefix('funcionario-payments')->name('funcionario-payments.')->group(function () {
            Route::get('/', [\App\Http\Controllers\Admin\FuncionarioServicoPaymentController::class, 'index'])->name('index');
            Route::post('/', [\App\Http\Controllers\Admin\FuncionarioServicoPaymentController::class, 'store'])->name('store');
            Route::put('/{payment}', [\App\Http\Controllers\Admin\FuncionarioServicoPaymentController::class, 'update'])->name('update');
            Route::post('/{payment}/marcar-pago', [\App\Http\Controllers\Admin\FuncionarioServicoPaymentController::class, 'marcarComoPago'])->name('marcar-pago');
            Route::post('/marcar-todos-pago', [\App\Http\Controllers\Admin\FuncionarioServicoPaymentController::class, 'marcarTodosComoPago'])->name('marcar-todos-pago');
            Route::get('/folha-salario', [\App\Http\Controllers\Admin\FuncionarioServicoPaymentController::class, 'gerarFolhaSalario'])->name('folha-salario');
            Route::get('/folha-salario/visualizar', [\App\Http\Controllers\Admin\FuncionarioServicoPaymentController::class, 'visualizarFolhaSalario'])->name('folha-salario.visualizar');
            Route::post('/gerar-mes-seguinte', [\App\Http\Controllers\Admin\FuncionarioServicoPaymentController::class, 'gerarMesSeguinte'])->name('gerar-mes-seguinte');
        });

        // Pagamentos da Administração (funcionários administrativos)
        Route::prefix('administracao-payments')->name('administracao-payments.')->group(function () {
            Route::get('/', [\App\Http\Controllers\Admin\AdministracaoPaymentController::class, 'index'])->name('index');
            Route::post('/', [\App\Http\Controllers\Admin\AdministracaoPaymentController::class, 'store'])->name('store');
            Route::put('/{payment}', [\App\Http\Controllers\Admin\AdministracaoPaymentController::class, 'update'])->name('update');
            Route::post('/{payment}/marcar-pago', [\App\Http\Controllers\Admin\AdministracaoPaymentController::class, 'marcarComoPago'])->name('marcar-pago');
            Route::post('/marcar-todos-pago', [\App\Http\Controllers\Admin\AdministracaoPaymentController::class, 'marcarTodosComoPago'])->name('marcar-todos-pago');
            Route::get('/folha-salario', [\App\Http\Controllers\Admin\AdministracaoPaymentController::class, 'gerarFolhaSalario'])->name('folha-salario');
            Route::get('/folha-salario/visualizar', [\App\Http\Controllers\Admin\AdministracaoPaymentController::class, 'visualizarFolhaSalario'])->name('folha-salario.visualizar');
            Route::post('/gerar-mes-seguinte', [\App\Http\Controllers\Admin\AdministracaoPaymentController::class, 'gerarMesSeguinte'])->name('gerar-mes-seguinte');
        });

        // Turmas
        Route::get('/turmas', [AdminController::class, 'indexTurmas'])->name('turmas.index');
        Route::get('/turmas/create', [AdminController::class, 'createTurma'])->name('turmas.create');
        Route::post('/turmas', [AdminController::class, 'storeTurma'])->name('turmas.store');
        Route::get('/turmas/{turma}', [AdminController::class, 'showTurma'])->name('turmas.show');
        Route::get('/turmas/{turma}/edit', [AdminController::class, 'editTurma'])->name('turmas.edit');
        Route::put('/turmas/{turma}', [AdminController::class, 'updateTurma'])->name('turmas.update');
        Route::delete('/turmas/{turma}', [AdminController::class, 'destroyTurma'])->name('turmas.destroy');
        Route::get('/turmas/{turma}/alocar-alunos', [AdminController::class, 'alocarAlunos'])->name('turmas.alocar-alunos');
        Route::post('/turmas/{turma}/alocar-alunos', [AdminController::class, 'storeAlocacaoAlunos'])->name('turmas.alocar-alunos.store');
        Route::delete('/turmas/{turma}/alunos/{aluno}', [AdminController::class, 'removerAluno'])->name('turmas.remover-aluno');
        Route::get('/turmas/{turma}/atribuir-disciplinas', [AdminController::class, 'atribuirDisciplinas'])->name('turmas.atribuir-disciplinas');
        Route::post('/turmas/{turma}/atribuir-disciplinas', [AdminController::class, 'storeAtribuicaoDisciplinas'])->name('turmas.atribuir-disciplinas.store');
        Route::delete('/turmas/{turma}/disciplinas/{disciplina}', [AdminController::class, 'removerDisciplina'])->name('turmas.remover-disciplina');

        // Disciplinas
        Route::get('/disciplinas', [AdminController::class, 'indexDisciplinas'])->name('disciplinas.index');
        Route::get('/disciplinas/create', [AdminController::class, 'createDisciplina'])->name('disciplinas.create');
        Route::post('/disciplinas', [AdminController::class, 'storeDisciplina'])->name('disciplinas.store');
        Route::get('/disciplinas/{disciplina}/edit', [AdminController::class, 'editDisciplina'])->name('disciplinas.edit');
        Route::put('/disciplinas/{disciplina}', [AdminController::class, 'updateDisciplina'])->name('disciplinas.update');
        Route::delete('/disciplinas/{disciplina}', [AdminController::class, 'destroyDisciplina'])->name('disciplinas.destroy');

        // Funcionários Administrativos
        Route::get('/funcionarios', [FuncionarioController::class, 'index'])->name('funcionarios.index');
        Route::get('/funcionarios/create', [FuncionarioController::class, 'create'])->name('funcionarios.create');
        Route::post('/funcionarios', [FuncionarioController::class, 'store'])->name('funcionarios.store');
        Route::get('/funcionarios/{funcionario}', [FuncionarioController::class, 'show'])->name('funcionarios.show');
        Route::get('/funcionarios/{funcionario}/edit', [FuncionarioController::class, 'edit'])->name('funcionarios.edit');
        Route::put('/funcionarios/{funcionario}', [FuncionarioController::class, 'update'])->name('funcionarios.update');
        Route::delete('/funcionarios/{funcionario}', [FuncionarioController::class, 'destroy'])->name('funcionarios.destroy');
        Route::patch('/funcionarios/{funcionario}/toggle-status', [FuncionarioController::class, 'toggleStatus'])->name('funcionarios.toggle-status');

        // Funcionários (Serviços: limpeza, segurança, etc.)
        Route::prefix('funcionarios-servicos')->name('funcionarios-servicos.')->group(function () {
            Route::get('/', [\App\Http\Controllers\Admin\FuncionarioServicoController::class, 'index'])->name('index');
            Route::get('/create', [\App\Http\Controllers\Admin\FuncionarioServicoController::class, 'create'])->name('create');
            Route::post('/', [\App\Http\Controllers\Admin\FuncionarioServicoController::class, 'store'])->name('store');
            Route::get('/{funcionarioServico}', [\App\Http\Controllers\Admin\FuncionarioServicoController::class, 'show'])->name('show');
            Route::get('/{funcionarioServico}/edit', [\App\Http\Controllers\Admin\FuncionarioServicoController::class, 'edit'])->name('edit');
            Route::put('/{funcionarioServico}', [\App\Http\Controllers\Admin\FuncionarioServicoController::class, 'update'])->name('update');
            Route::delete('/{funcionarioServico}', [\App\Http\Controllers\Admin\FuncionarioServicoController::class, 'destroy'])->name('destroy');
            Route::patch('/{funcionarioServico}/toggle-status', [\App\Http\Controllers\Admin\FuncionarioServicoController::class, 'toggleStatus'])->name('toggle-status');
        });

        // Recuperação de Senha
        Route::get('/password-reset', [PasswordResetController::class, 'index'])->name('password-reset.index');
        Route::get('/password-reset/{user}', [PasswordResetController::class, 'show'])->name('password-reset.show');
        Route::post('/password-reset/{user}', [PasswordResetController::class, 'reset'])->name('password-reset.reset');

        // Verificação de Facturas
        Route::get('/invoice/verificar', [App\Http\Controllers\InvoiceController::class, 'verificar'])->name('invoice.verificar');
    });

    // Avaliações
    Route::get('/avaliacoes', [AvaliacaoController::class, 'index'])->name('avaliacoes.index');
    Route::get('/avaliacoes/create', [AvaliacaoController::class, 'create'])->name('avaliacoes.create');
    Route::post('/avaliacoes', [AvaliacaoController::class, 'store'])->name('avaliacoes.store');
    Route::get('/avaliacoes/{avaliacao}/edit', [AvaliacaoController::class, 'edit'])->name('avaliacoes.edit');
    Route::put('/avaliacoes/{avaliacao}', [AvaliacaoController::class, 'update'])->name('avaliacoes.update');
    Route::get('/avaliacoes/imprimir-ficha', [AvaliacaoController::class, 'imprimirFicha'])->name('avaliacoes.imprimir-ficha');
    Route::get('/avaliacoes/todas-disciplinas', [AvaliacaoController::class, 'todasDisciplinas'])->name('avaliacoes.todas-disciplinas');
    Route::get('/avaliacoes/exportar-excel', [AvaliacaoController::class, 'exportarExcel'])->name('avaliacoes.exportar-excel');
    Route::get('/avaliacoes/download-pdf-todas-disciplinas', [AvaliacaoController::class, 'downloadPdfTodasDisciplinas'])->name('avaliacoes.download-pdf-todas-disciplinas');
    Route::get('/avaliacoes/notas-por-estudante', [AvaliacaoController::class, 'notasPorEstudante'])->name('avaliacoes.notas-por-estudante');
    Route::get('/avaliacoes/imprimir-notas-por-estudante', [AvaliacaoController::class, 'imprimirNotasPorEstudante'])->name('avaliacoes.imprimir-notas-por-estudante');
    Route::get('/avaliacoes/download-pdf-notas-por-estudante', [AvaliacaoController::class, 'downloadPdfNotasPorEstudante'])->name('avaliacoes.download-pdf-notas-por-estudante');
    Route::get('/avaliacoes/estatisticas', [AvaliacaoController::class, 'estatisticas'])->name('avaliacoes.estatisticas');
    Route::get('/avaliacoes/situacao-pedagogica', [AvaliacaoController::class, 'situacaoPedagogica'])->name('avaliacoes.situacao-pedagogica');

    // Financeiro
    Route::get('/financeiro', [FinanceiroController::class, 'index'])->name('financeiro.index');
    Route::get('/financeiro/create', [FinanceiroController::class, 'create'])->name('financeiro.create');
    Route::post('/financeiro', [FinanceiroController::class, 'store'])->name('financeiro.store');
        Route::post('/financeiro/{financeiro}/pagar', [FinanceiroController::class, 'pagar'])->name('financeiro.pagar');
        Route::post('/financeiro/pagar-aluno-mes', [FinanceiroController::class, 'pagarAlunoMes'])->name('financeiro.pagar-aluno-mes');
        Route::get('/financeiro/verificar-gateway/{financeiro}', [FinanceiroController::class, 'verificarGateway'])->name('financeiro.verificar-gateway');
        Route::get('/financeiro/buscar-alunos', [FinanceiroController::class, 'buscarAlunos'])->name('financeiro.buscar-alunos');
        Route::get('/financeiro/pagamento-sucesso/{invoice}', [FinanceiroController::class, 'pagamentoSucesso'])->name('financeiro.pagamento-sucesso');
        Route::get('/financeiro/factura/{invoice}/download', [FinanceiroController::class, 'downloadFactura'])->name('financeiro.factura.download');
        Route::get('/financeiro/configurar-mensalidades', [FinanceiroController::class, 'configurarMensalidadesIndex'])->name('financeiro.configurar-mensalidades');
        Route::post('/financeiro/configurar-mensalidades', [FinanceiroController::class, 'configurarMensalidades'])->name('financeiro.configurar-mensalidades.store');
        Route::get('/financeiro/aluno/{aluno}/facturas', [FinanceiroController::class, 'facturasAluno'])->name('financeiro.facturas-aluno');
        Route::get('/financeiro/factura/{invoice}/download-admin', [FinanceiroController::class, 'downloadFacturaAluno'])->name('financeiro.factura.download-admin');
        Route::get('/financeiro/factura/{invoice}/visualizar', [FinanceiroController::class, 'visualizarFactura'])->name('financeiro.factura.visualizar');
        Route::get('/financeiro/relatorio', [FinanceiroController::class, 'relatorioFinanceiro'])->name('financeiro.relatorio');
        Route::get('/financeiro/gerar-mensalidades', [FinanceiroController::class, 'criarMensalidades'])->name('financeiro.gerar-mensalidades');
        Route::post('/financeiro/gerar-mensalidades', [FinanceiroController::class, 'gerarMensalidades'])->name('financeiro.gerar-mensalidades.store');
        Route::post('/financeiro/bloquear-alunos-pendentes', [FinanceiroController::class, 'bloquearAlunosPendentes'])->name('financeiro.bloquear-alunos-pendentes');

    // Trabalhos / Planos
    Route::get('/trabalhos', [TrabalhoController::class, 'index'])->name('trabalhos.index');
    Route::get('/trabalhos/create', [TrabalhoController::class, 'create'])->name('trabalhos.create');
    Route::post('/trabalhos', [TrabalhoController::class, 'store'])->name('trabalhos.store');
    Route::get('/planos/{plano}/visualizar', [TrabalhoController::class, 'visualizarPlano'])->name('planos.visualizar');

    // Horários
    Route::get('/horarios', [HorarioController::class, 'index'])->name('horarios.index');
    Route::get('/horarios/create', [HorarioController::class, 'create'])->name('horarios.create');
    Route::post('/horarios', [HorarioController::class, 'store'])->name('horarios.store');
    Route::get('/horarios/download-pdf', [HorarioController::class, 'downloadPdf'])->name('horarios.download-pdf');
    Route::delete('/horarios/eliminar-todos', [HorarioController::class, 'destroyAll'])->name('horarios.destroy-all');
    Route::get('/horarios/gerar-automatico', [HorarioController::class, 'gerarAutomatico'])->name('horarios.gerar-automatico');
    Route::post('/horarios/gerar-automatico', [HorarioController::class, 'storeAutomatico'])->name('horarios.store-automatico');
    Route::get('/horarios/gerar-todas-turmas', [HorarioController::class, 'gerarTodasTurmasView'])->name('horarios.gerar-todas-turmas');
    Route::post('/horarios/gerar-todas-turmas', [HorarioController::class, 'gerarTodasTurmas'])->name('horarios.store-todas-turmas');
    Route::get('/horarios/{horario}/edit', [HorarioController::class, 'edit'])->name('horarios.edit');
    Route::put('/horarios/{horario}', [HorarioController::class, 'update'])->name('horarios.update');
    Route::delete('/horarios/{horario}', [HorarioController::class, 'destroy'])->name('horarios.destroy');

    // Eventos
    Route::get('/eventos', [EventoController::class, 'index'])->name('eventos.index');
    Route::get('/eventos/create', [EventoController::class, 'create'])->name('eventos.create');
    Route::post('/eventos', [EventoController::class, 'store'])->name('eventos.store');
    Route::get('/eventos/{evento}/edit', [EventoController::class, 'edit'])->name('eventos.edit');
    Route::put('/eventos/{evento}', [EventoController::class, 'update'])->name('eventos.update');
    Route::delete('/eventos/{evento}', [EventoController::class, 'destroy'])->name('eventos.destroy');
    Route::patch('/eventos/{evento}/cancelar', [EventoController::class, 'cancelar'])->name('eventos.cancelar');

    // Área do Professor
    Route::prefix('professor')->middleware(['role:professor'])->name('professor.')->group(function () {
        Route::get('/dashboard', [ProfessorController::class, 'dashboard'])->name('dashboard');
        Route::get('/horarios', [ProfessorController::class, 'horarios'])->name('horarios');
        Route::get('/turmas', [ProfessorController::class, 'turmas'])->name('turmas');
        Route::get('/perfil', [ProfessorController::class, 'perfil'])->name('perfil');
        
        // TPCs
        Route::get('/tpcs', [TPCPlanoController::class, 'indexTPCs'])->name('tpcs.index');
        Route::get('/tpcs/create', [TPCPlanoController::class, 'createTPC'])->name('tpcs.create');
        Route::post('/tpcs', [TPCPlanoController::class, 'storeTPC'])->name('tpcs.store');
        Route::get('/tpcs/{tpc}', [TPCPlanoController::class, 'showTPC'])->name('tpcs.show');
        Route::get('/tpcs/{tpc}/edit', [TPCPlanoController::class, 'editTPC'])->name('tpcs.edit');
        Route::put('/tpcs/{tpc}', [TPCPlanoController::class, 'updateTPC'])->name('tpcs.update');
        Route::delete('/tpcs/{tpc}', [TPCPlanoController::class, 'destroyTPC'])->name('tpcs.destroy');
        Route::get('/tpcs/{tpc}/visualizar', [TPCPlanoController::class, 'visualizarTPC'])->name('tpcs.visualizar');
        
        // Planos Trimestrais
        Route::get('/planos', [TPCPlanoController::class, 'indexPlanos'])->name('planos.index');
        Route::get('/planos/create', [TPCPlanoController::class, 'createPlano'])->name('planos.create');
        Route::post('/planos', [TPCPlanoController::class, 'storePlano'])->name('planos.store');
        Route::get('/planos/{plano}', [TPCPlanoController::class, 'showPlano'])->name('planos.show');
        Route::get('/planos/{plano}/edit', [TPCPlanoController::class, 'editPlano'])->name('planos.edit');
        Route::put('/planos/{plano}', [TPCPlanoController::class, 'updatePlano'])->name('planos.update');
        Route::delete('/planos/{plano}', [TPCPlanoController::class, 'destroyPlano'])->name('planos.destroy');
        Route::get('/planos/{plano}/visualizar', [TPCPlanoController::class, 'visualizarPlano'])->name('planos.visualizar');
        Route::get('/planos/{plano}/download', [TPCPlanoController::class, 'downloadPlano'])->name('planos.download');
    });

    // Área do Estudante
    Route::prefix('aluno')->middleware(['role:aluno', 'aluno.bloqueado'])->name('aluno.')->group(function () {
        Route::get('/dashboard', [AlunoController::class, 'dashboard'])->name('dashboard');
        Route::get('/notas', [AlunoController::class, 'notas'])->name('notas');
        Route::get('/folheto-aproveitamento', [AlunoController::class, 'folhetoAproveitamento'])->name('folheto-aproveitamento');
        Route::get('/horarios', [AlunoController::class, 'horarios'])->name('horarios');
        Route::get('/trabalhos', [AlunoController::class, 'trabalhos'])->name('trabalhos');
        Route::get('/trabalhos/{trabalho}/visualizar', [AlunoController::class, 'visualizarTrabalho'])->name('trabalhos.visualizar');
        Route::get('/eventos', [AlunoController::class, 'eventos'])->name('eventos');
        Route::get('/financeiro', [AlunoController::class, 'financeiro'])->name('financeiro');
        Route::get('/perfil', [AlunoController::class, 'perfil'])->name('perfil');

        // Pagamentos
        Route::get('/payment/create', [App\Http\Controllers\PaymentController::class, 'create'])->name('payment.create');
        Route::post('/payment', [App\Http\Controllers\PaymentController::class, 'store'])->name('payment.store');
        Route::get('/payment/{payment}/process', [App\Http\Controllers\PaymentController::class, 'process'])->name('payment.process');
        Route::post('/payment/{payment}/confirm', [App\Http\Controllers\PaymentController::class, 'confirm'])->name('payment.confirm');
        Route::get('/payment/{payment}/success', [App\Http\Controllers\PaymentController::class, 'success'])->name('payment.success');
        Route::post('/payment/callback', [App\Http\Controllers\PaymentController::class, 'callback'])->name('payment.callback');

        // Facturas
        Route::get('/invoice', [App\Http\Controllers\InvoiceController::class, 'index'])->name('invoice.index');
        Route::get('/invoice/{invoice}', [App\Http\Controllers\InvoiceController::class, 'show'])->name('invoice.show');
        Route::get('/invoice/{invoice}/download', [App\Http\Controllers\InvoiceController::class, 'download'])->name('invoice.download');
    });

    // Chat e Notificações (acessíveis por todos os usuários autenticados)
    Route::prefix('chat')->name('chat.')->group(function () {
        Route::get('/', [ChatController::class, 'index'])->name('index');
        Route::get('/create', [ChatController::class, 'create'])->name('create');
        Route::post('/', [ChatController::class, 'store'])->name('store');
        Route::post('/mass', [ChatController::class, 'storeMass'])->name('store-mass');
        Route::get('/unread/count', [ChatController::class, 'unreadCount'])->name('unread.count');
        Route::post('/typing', [ChatController::class, 'typing'])->name('typing');
        Route::get('/typing/check', [ChatController::class, 'checkTyping'])->name('typing.check');
        Route::get('/new-messages', [ChatController::class, 'getNewMessages'])->name('new-messages');
        Route::post('/mark-as-read', [ChatController::class, 'markAsRead'])->name('mark-as-read');
        Route::delete('/message/{message}', [ChatController::class, 'destroy'])->name('destroy');
        Route::post('/message/{message}/delete', [ChatController::class, 'destroy'])->name('destroy.post');
        Route::post('/delete-conversation', [ChatController::class, 'deleteConversation'])->name('delete-conversation');
        Route::get('/{user}', [ChatController::class, 'show'])->name('show');
    });

    Route::prefix('notifications')->name('notifications.')->group(function () {
        Route::get('/', [NotificationController::class, 'index'])->name('index');
        Route::get('/unread', [NotificationController::class, 'unread'])->name('unread');
        Route::post('/{notification}/read', [NotificationController::class, 'markAsRead'])->name('read');
        Route::post('/read-all', [NotificationController::class, 'markAllAsRead'])->name('read-all');
    });
});

